<?php
define('IS_AJAX', true);
require_once '../administracion/includes/seguridad.php';

header('Content-Type: application/json');
$response = ['success' => false, 'message' => 'Acción no válida.'];
$usuario_id = $_SESSION['user_id'] ?? 0;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        $response['message'] = 'Error de seguridad (CSRF).';
        echo json_encode($response);
        exit();
    }
}

$action = $_POST['action'] ?? $_GET['action'] ?? '';

if (!puede('gestionar_clientes_admin')) {
    $response['message'] = 'Acceso denegado.';
    echo json_encode($response);
    exit();
}

/**
 * Verifica si un número tiene el formato de un celular venezolano.
 */
function es_telefono_venezolano($telefono) {
    $numeros = preg_replace('/[^0-9]/', '', $telefono);
    $len = strlen($numeros);

    // Formato: 584XX1234567 (12 dígitos)
    if ($len === 12 && strpos($numeros, '584') === 0) return true;
    // Formato: 04XX1234567 (11 dígitos)
    if ($len === 11 && strpos($numeros, '04') === 0) return true;
    // Formato: 4XX1234567 (10 dígitos)
    if ($len === 10 && strpos($numeros, '4') === 0) return true;
    
    return false;
}

/**
 * Normaliza un número de teléfono venezolano al formato estándar 584...
 */
function normalizar_telefono_venezolano($telefono) {
    if (empty($telefono)) return '';
    $numeros = preg_replace('/[^0-9]/', '', $telefono);
    $len = strlen($numeros);

    if ($len === 12 && strpos($numeros, '584') === 0) return $numeros;
    if ($len === 11 && strpos($numeros, '04') === 0) return '58' . substr($numeros, 1);
    if ($len === 10 && strpos($numeros, '4') === 0) return '58' . $numeros;
    
    return $numeros; // Devuelve el número limpio si no coincide con un formato esperado
}

/**
 * Verifica si un teléfono venezolano ya existe en la BD.
 */
function telefono_existe($conn, $telefono) {
    $telefono_normalizado = normalizar_telefono_venezolano($telefono);
    if (empty($telefono_normalizado)) return true;

    // Busca de 3 formas: 58414..., 414..., y 0414...
    $stmt = $conn->prepare("SELECT COUNT(*) FROM clientes WHERE 
        REPLACE(REPLACE(REPLACE(telefono, ' ', ''), '-', ''), '+', '') LIKE ? OR
        REPLACE(REPLACE(REPLACE(telefono, ' ', ''), '-', ''), '+', '') LIKE ? OR
        REPLACE(REPLACE(REPLACE(telefono, ' ', ''), '-', ''), '+', '') LIKE ?");
    
    $telefono_sin_prefijo = substr($telefono_normalizado, 2);
    $stmt->execute([
        "%" . $telefono_normalizado,
        "%" . $telefono_sin_prefijo,
        "%0" . $telefono_sin_prefijo
    ]);
    return $stmt->fetchColumn() > 0;
}

/**
 * Valida los datos del cliente, asegurando que el teléfono sea venezolano.
 */
function validar_cliente_venezolano($cliente_data) {
    $errores = [];
    if (empty($cliente_data['telefono'])) {
        $errores[] = 'Teléfono es obligatorio';
    } elseif (!es_telefono_venezolano($cliente_data['telefono'])) {
        // Este caso se maneja en el bucle principal, pero es una doble seguridad.
        $errores[] = 'El teléfono no tiene un formato venezolano válido.';
    }
    return ['valido' => empty($errores), 'errores' => $errores];
}

/**
 * Inserta un cliente en la base de datos (se asume que el teléfono ya fue validado y normalizado).
 */
function insertar_cliente($conn, $cliente_data, $usuario_id) {
    $sql = "INSERT INTO clientes (nombre, telefono, ciudad, origen, estado, estado_global, fecha_primer_contacto, creado_por_id) VALUES (?, ?, ?, ?, ?, ?, NOW(), ?)";
    
    $nombre = limpiar_nombre($cliente_data['nombre'] ?? '');
    // Importante: Normaliza el teléfono justo antes de insertar.
    $telefono = normalizar_telefono_venezolano($cliente_data['telefono'] ?? '');
    $ciudad = limpiar_texto($cliente_data['ciudad'] ?? '');
    $origen = limpiar_texto($cliente_data['origen'] ?? '') ?: 'importacion';
    
    $stmt = $conn->prepare($sql);
    return $stmt->execute([$nombre, $telefono, $ciudad, $origen, 'nuevo', 'Importado desde Excel', $usuario_id]);
}

// Las funciones limpiar_nombre, limpiar_texto, cedula_existe y procesar_fecha se mantienen igual que en el código anterior...
function limpiar_nombre($nombre) { if (empty(trim($nombre))) return 'Sin Nombre'; $nombre = preg_replace('/[\x{1F300}-\x{1F5FF}\x{1F900}-\x{1F9FF}\x{1F600}-\x{1F64F}\x{1F680}-\x{1F6FF}\x{2600}-\x{26FF}\x{2700}-\x{27BF}\x{1F1E6}-\x{1F1FF}]/u', '', $nombre); $nombre = preg_replace('/[^\p{L}\p{N}\s\.\-]/u', '', $nombre); $nombre = preg_replace('/\s+/', ' ', trim($nombre)); $nombre = mb_convert_case($nombre, MB_CASE_TITLE, 'UTF-8'); return !empty($nombre) ? $nombre : 'Sin Nombre'; }
function limpiar_texto($texto) { if (empty($texto)) return ''; $texto = preg_replace('/[\x{1F300}-\x{1F5FF}\x{1F900}-\x{1F9FF}\x{1F600}-\x{1F64F}\x{1F680}-\x{1F6FF}\x{2600}-\x{26FF}\x{2700}-\x{27BF}\x{1F1E6}-\x{1F1FF}]/u', '', $texto); return preg_replace('/\s+/', ' ', trim($texto)); }
function cedula_existe($conn, $cedula) { if (empty($cedula)) return false; $stmt = $conn->prepare("SELECT COUNT(*) FROM clientes WHERE cedula = ?"); $stmt->execute([$cedula]); return $stmt->fetchColumn() > 0; }
function procesar_fecha($fecha_str) { if (empty($fecha_str)) return null; $formatos = ['Y-m-d', 'd/m/Y', 'm/d/Y', 'd-m-Y', 'm-d-Y', 'Y/m/d']; foreach ($formatos as $formato) { $fecha = DateTime::createFromFormat($formato, $fecha_str); if ($fecha !== false) return $fecha->format('Y-m-d H:i:s'); } if (is_numeric($fecha_str)) { try { $fecha_excel = DateTime::createFromFormat('U', strtotime('1900-01-01') + ($fecha_str - 2) * 86400); if ($fecha_excel !== false) return $fecha_excel->format('Y-m-d H:i:s'); } catch (Exception $e) {} } return null; }


switch ($action) {
    case 'importar_clientes_lote':
        try {
            $clientes = json_decode($_POST['clientes'] ?? '[]', true);
            if (!is_array($clientes)) {
                $response['message'] = 'Formato de datos inválido.';
                break;
            }

            // Añadimos contadores para los omitidos
            $stats = ['processed' => 0, 'success' => 0, 'duplicates' => 0, 'errors' => 0, 'omitted' => 0, 'duplicatesList' => [], 'omittedList' => []];
            
            $conn->beginTransaction();

            foreach ($clientes as $cliente_data) {
                $stats['processed']++;
                $telefono_original = $cliente_data['telefono'] ?? '';

                if (empty(trim($telefono_original))) {
                    $stats['errors']++;
                    continue; // Si no hay teléfono, se cuenta como error.
                }

                // *** LÓGICA PRINCIPAL: VERIFICAR SI ES VENEZOLANO ***
                if (es_telefono_venezolano($telefono_original)) {
                    // Si es venezolano, procede con la validación y verificación de duplicados.
                    if (telefono_existe($conn, $telefono_original)) {
                        $stats['duplicates']++;
                        $stats['duplicatesList'][] = ['nombre' => limpiar_nombre($cliente_data['nombre'] ?? ''), 'telefono' => $telefono_original];
                    } else {
                        if (insertar_cliente($conn, $cliente_data, $usuario_id)) {
                            $stats['success']++;
                        } else {
                            $stats['errors']++;
                        }
                    }
                } else {
                    // Si NO es venezolano, se omite y se registra.
                    $stats['omitted']++;
                    $stats['omittedList'][] = [
                        'nombre' => limpiar_nombre($cliente_data['nombre'] ?? ''),
                        'telefono' => $telefono_original
                    ];
                }
            }

            $conn->commit();
            
            if (function_exists('registrar_accion')) {
                registrar_accion('importar_clientes_excel', "Importe: {$stats['success']} exitosos, {$stats['duplicates']} duplicados, {$stats['omitted']} omitidos, {$stats['errors']} errores.");
            }
            $response = ['success' => true, 'message' => 'Lote procesado.', 'stats' => $stats];

        } catch (Exception $e) {
            $conn->rollBack();
            $response['message'] = 'Error al procesar lote: ' . $e->getMessage();
            error_log("Error en importar_clientes_lote: " . $e->getMessage());
        }
        break;

    // El 'case' para verificar duplicados también se ajusta a la lógica venezolana
    case 'verificar_duplicados_previo':
        try {
            $telefonos = json_decode($_POST['telefonos'] ?? '[]', true);
            if (!is_array($telefonos)) {
                $response['message'] = 'Formato de datos inválido.';
                break;
            }
            $duplicados = [];
            foreach ($telefonos as $telefono) {
                // Solo verifica duplicados si es un número venezolano
                if (es_telefono_venezolano($telefono) && telefono_existe($conn, $telefono)) {
                    $duplicados[] = normalizar_telefono_venezolano($telefono);
                }
            }
            $response = ['success' => true, 'duplicados' => array_unique($duplicados), 'total_verificados' => count($telefonos), 'total_duplicados' => count(array_unique($duplicados))];
        } catch (Exception $e) {
            $response['message'] = 'Error al verificar duplicados: ' . $e->getMessage();
        }
        break;

    default:
        $response['message'] = 'Acción desconocida.';
        break;
}

echo json_encode($response);