<?php
require_once '../administracion/includes/seguridad.php';

// Verificar permisos
if (!puede('gestionar_clientes_admin')) {
    header('Location: ../login.php');
    exit();
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Importar Clientes - Sistema</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.3.0/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <style>
        .upload-area {
            border: 2px dashed #dee2e6;
            border-radius: 10px;
            padding: 40px;
            text-align: center;
            transition: all 0.3s ease;
            background: #f8f9fa;
            cursor: pointer;
        }
        .upload-area:hover {
            border-color: #0d6efd;
            background: #e7f3ff;
        }
        .upload-area.dragover {
            border-color: #0d6efd;
            background: #e7f3ff;
            transform: scale(1.02);
        }
        .progress-container {
            display: none;
        }
        .sheet-selector {
            display: none;
            margin-top: 20px;
        }
        .preview-container {
            display: none;
            margin-top: 20px;
        }
        .results-container {
            display: none;
            margin-top: 20px;
        }
        .file-info {
            background: #e9ecef;
            padding: 15px;
            border-radius: 8px;
            margin-top: 15px;
        }
        .table-responsive {
            max-height: 400px;
            overflow-y: auto;
        }
        .duplicados-list {
            max-height: 300px;
            overflow-y: auto;
        }
        .step-indicator {
            display: flex;
            justify-content: center;
            margin-bottom: 30px;
        }
        .step {
            display: flex;
            align-items: center;
            padding: 0 20px;
        }
        .step-circle {
            width: 30px;
            height: 30px;
            border-radius: 50%;
            background: #dee2e6;
            color: #6c757d;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 10px;
        }
        .step.active .step-circle {
            background: #0d6efd;
            color: white;
        }
        .step.completed .step-circle {
            background: #198754;
            color: white;
        }
        .multi-sheet-info {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px;
            border-radius: 10px;
            margin: 20px 0;
        }
        .cleaned-data-preview {
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 8px;
            padding: 15px;
            margin: 10px 0;
        }
        .data-cleaning-info {
            background: #e7f3ff;
            border: 1px solid #b3d7ff;
            border-radius: 8px;
            padding: 15px;
            margin: 15px 0;
        }
        .debug-panel {
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 8px;
            padding: 15px;
            margin: 20px 0;
            font-family: monospace;
            font-size: 12px;
            max-height: 200px;
            overflow-y: auto;
        }
    </style>
</head>
<body>
    <script>
    const CSRF_TOKEN = '<?php echo $_SESSION['csrf_token']; ?>';
    </script>
    <div class="container-fluid py-4">
        <div class="row">
            <div class="col-12">
                <!-- Panel de Debug -->
                <div class="alert alert-info">
                    <h6><i class="fas fa-bug me-2"></i>Panel de Debug</h6>
                    <div class="debug-panel" id="debugLog">
                        Esperando eventos...
                    </div>
                </div>

                <div class="card shadow-sm">
                    <div class="card-header bg-primary text-white">
                        <h4 class="mb-0">
                            <i class="fas fa-file-excel me-2"></i>
                            Importar Clientes desde Excel
                        </h4>
                        <small>Procesa automáticamente todas las hojas del archivo</small>
                    </div>
                    <div class="card-body">
                        <!-- Indicador de pasos -->
                        <div class="step-indicator">
                            <div class="step active" id="step1">
                                <div class="step-circle">1</div>
                                <span>Seleccionar Archivo</span>
                            </div>
                            <div class="step" id="step2">
                                <div class="step-circle">2</div>
                                <span>Análisis</span>
                            </div>
                            <div class="step" id="step3">
                                <div class="step-circle">3</div>
                                <span>Vista Previa</span>
                            </div>
                            <div class="step" id="step4">
                                <div class="step-circle">4</div>
                                <span>Importar</span>
                            </div>
                            <div class="step" id="step5">
                                <div class="step-circle">5</div>
                                <span>Resultados</span>
                            </div>
                        </div>

                        <!-- Información sobre limpieza de datos -->
                        <div class="data-cleaning-info">
                            <h6><i class="fas fa-magic me-2"></i>Limpieza Automática de Datos</h6>
                            <div class="row">
                                <div class="col-md-4">
                                    <strong>Nombres:</strong>
                                    <ul class="small mb-0">
                                        <li>Elimina emojis y caracteres especiales</li>
                                        <li>Si está vacío: "Sin Nombre"</li>
                                        <li>Capitaliza correctamente</li>
                                    </ul>
                                </div>
                                <div class="col-md-4">
                                    <strong>Teléfonos:</strong>
                                    <ul class="small mb-0">
                                        <li><code>4144533455</code> → <code>584144533455</code></li>
                                        <li><code>414-434343</code> → <code>58414434343</code></li>
                                        <li><code>04125435425</code> → <code>584125435425</code></li>
                                    </ul>
                                </div>
                                <div class="col-md-4">
                                    <strong>Textos:</strong>
                                    <ul class="small mb-0">
                                        <li>Elimina emojis</li>
                                        <li>Limpia espacios múltiples</li>
                                        <li>Normaliza formato</li>
                                    </ul>
                                </div>
                            </div>
                        </div>

                        <!-- Área de carga de archivo -->
                        <div id="upload-section">
                            <div class="upload-area" id="uploadArea">
                                <i class="fas fa-cloud-upload-alt fa-3x text-muted mb-3"></i>
                                <h5>Arrastra tu archivo Excel aquí</h5>
                                <p class="text-muted">Se procesarán automáticamente todas las hojas del archivo</p>
                                <input type="file" id="fileInput" class="d-none" accept=".xlsx,.xls">
                                <button type="button" class="btn btn-outline-primary" id="selectFileBtn">
                                    <i class="fas fa-folder-open me-2"></i>Seleccionar Archivo Excel
                                </button>
                                <small class="d-block mt-2 text-muted">Tamaño máximo: 50MB | Formatos: .xlsx, .xls</small>
                            </div>
                            
                            <div class="file-info" id="fileInfo" style="display: none;">
                                <div class="row">
                                    <div class="col-md-4">
                                        <strong>Archivo:</strong> <span id="fileName"></span>
                                    </div>
                                    <div class="col-md-2">
                                        <strong>Tamaño:</strong> <span id="fileSize"></span>
                                    </div>
                                    <div class="col-md-2">
                                        <strong>Hojas:</strong> <span id="sheetsCount">Analizando...</span>
                                    </div>
                                    <div class="col-md-4">
                                        <strong>Estado:</strong> <span id="fileStatus" class="badge bg-info">Procesando...</span>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Información de múltiples hojas -->
                        <div class="multi-sheet-info" id="multiSheetInfo" style="display: none;">
                            <div class="row align-items-center">
                                <div class="col-md-8">
                                    <h5 class="mb-1"><i class="fas fa-layers me-2"></i>Archivo con Múltiples Hojas Detectado</h5>
                                    <p class="mb-0">Se procesarán automáticamente todas las hojas del archivo. Los datos se combinarán en una sola importación.</p>
                                </div>
                                <div class="col-md-4 text-center">
                                    <div class="h2 mb-0" id="totalRecordsCount">0</div>
                                    <small>Registros totales encontrados</small>
                                </div>
                            </div>
                        </div>

                        <!-- Selector de hoja (solo para archivo de una hoja) -->
                        <div class="sheet-selector" id="sheetSelector">
                            <div class="row">
                                <div class="col-md-6">
                                    <label for="sheetSelect" class="form-label">Seleccionar Hoja de Excel:</label>
                                    <select class="form-select" id="sheetSelect">
                                        <option value="">Seleccione una hoja...</option>
                                    </select>
                                </div>
                                <div class="col-md-3">
                                    <label for="startRow" class="form-label">Fila de inicio:</label>
                                    <input type="number" class="form-control" id="startRow" value="2" min="1" max="100">
                                    <small class="text-muted">Fila donde empiezan los datos</small>
                                </div>
                                <div class="col-md-3 d-flex align-items-end">
                                    <button type="button" class="btn btn-info w-100" id="previewBtn" disabled>
                                        <i class="fas fa-eye me-2"></i>Vista Previa
                                    </button>
                                </div>
                            </div>
                        </div>

                        <!-- Vista previa -->
                        <div class="preview-container" id="previewContainer">
                            <div class="d-flex justify-content-between align-items-center mb-3">
                                <h5><i class="fas fa-table me-2"></i>Vista Previa de Datos</h5>
                                <div>
                                    <button type="button" class="btn btn-success" id="importBtn">
                                        <i class="fas fa-download me-2"></i>Iniciar Importación
                                    </button>
                                </div>
                            </div>
                            
                            <div class="alert alert-info">
                                <strong><i class="fas fa-map me-2"></i>Mapeo de Columnas:</strong>
                                <div class="row mt-2">
                                    <div class="col-md-2">
                                        <label class="form-label small">NOMBRE</label>
                                        <select class="form-select form-select-sm" id="mapNombre">
                                            <option value="">No mapear</option>
                                        </select>
                                    </div>
                                    <div class="col-md-2">
                                        <label class="form-label small">TELÉFONO *</label>
                                        <select class="form-select form-select-sm" id="mapTelefono">
                                            <option value="">Seleccionar</option>
                                        </select>
                                    </div>
                                    <div class="col-md-2">
                                        <label class="form-label small">CIUDAD</label>
                                        <select class="form-select form-select-sm" id="mapCiudad">
                                            <option value="">No mapear</option>
                                        </select>
                                    </div>
                                    <div class="col-md-2">
                                        <label class="form-label small">CAMPAÑA</label>
                                        <select class="form-select form-select-sm" id="mapOrigen">
                                            <option value="">No mapear</option>
                                        </select>
                                    </div>
                                    <div class="col-md-2">
                                        <label class="form-label small">FECHA</label>
                                        <select class="form-select form-select-sm" id="mapFecha">
                                            <option value="">No mapear</option>
                                        </select>
                                    </div>
                                    <div class="col-md-2 d-flex align-items-end">
                                        <button type="button" class="btn btn-sm btn-outline-secondary w-100" onclick="actualizarVistaPrevia()">
                                            <i class="fas fa-sync-alt me-1"></i>Actualizar
                                        </button>
                                    </div>
                                </div>
                                <small class="text-muted">* Campo obligatorio</small>
                            </div>

                            <!-- Preview de datos limpios -->
                            <div class="cleaned-data-preview" id="cleanedDataPreview" style="display: none;">
                                <h6><i class="fas fa-sparkles me-2"></i>Ejemplo de datos después de la limpieza:</h6>
                                <div class="row">
                                    <div class="col-md-6">
                                        <strong>Antes:</strong>
                                        <div class="small font-monospace text-muted" id="dataBefore"></div>
                                    </div>
                                    <div class="col-md-6">
                                        <strong>Después:</strong>
                                        <div class="small font-monospace text-success" id="dataAfter"></div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="table-responsive">
                                <table class="table table-sm table-striped" id="previewTable">
                                    <thead class="table-dark">
                                        <tr id="previewHeaders"></tr>
                                    </thead>
                                    <tbody id="previewBody"></tbody>
                                </table>
                            </div>
                        </div>

                        <!-- Barra de progreso -->
                        <div class="progress-container" id="progressContainer">
                            <div class="row">
                                <div class="col-md-8">
                                    <h5><i class="fas fa-cogs me-2"></i>Importando y limpiando datos...</h5>
                                    <div class="progress mb-3" style="height: 30px;">
                                        <div class="progress-bar progress-bar-striped progress-bar-animated bg-success" 
                                             id="progressBar" role="progressbar" style="width: 0%">
                                            <span id="progressText" class="fw-bold">0%</span>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="card text-center">
                                        <div class="card-body p-3">
                                            <div class="row">
                                                <div class="col-3">
                                                    <small class="text-muted">Procesados</small>
                                                    <div class="h5" id="processedCount">0</div>
                                                </div>
                                                <div class="col-3">
                                                    <small class="text-success">Exitosos</small>
                                                    <div class="h5 text-success" id="successCount">0</div>
                                                </div>
                                                <div class="col-3">
                                                    <small class="text-warning">Duplicados</small>
                                                    <div class="h5 text-warning" id="duplicateCount">0</div>
                                                </div>
                                                <div class="col-3">
                                                    <small class="text-danger">Errores</small>
                                                    <div class="h5 text-danger" id="errorCount">0</div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div id="statusMessage" class="alert alert-info">
                                <i class="fas fa-spinner fa-spin me-2"></i>Preparando importación...
                            </div>
                        </div>

                        <!-- Resultados -->
                        <div class="results-container" id="resultsContainer">
                            <div class="alert alert-success">
                                <h5><i class="fas fa-check-circle me-2"></i>¡Importación Completada Exitosamente!</h5>
                                <div class="row">
                                    <div class="col-md-3 text-center">
                                        <div class="h3 text-primary" id="finalProcessed">0</div>
                                        <small>Total Procesados</small>
                                    </div>
                                    <div class="col-md-3 text-center">
                                        <div class="h3 text-success" id="finalSuccess">0</div>
                                        <small>Importados Exitosamente</small>
                                    </div>
                                    <div class="col-md-3 text-center">
                                        <div class="h3 text-warning" id="finalDuplicates">0</div>
                                        <small>Duplicados Omitidos</small>
                                    </div>
                                    <div class="col-md-3 text-center">
                                        <div class="h3 text-danger" id="finalErrors">0</div>
                                        <small>Errores</small>
                                    </div>
                                </div>
                            </div>

                            <!-- Lista de duplicados -->
                            <div id="duplicatesSection" style="display: none;">
                                <h6><i class="fas fa-exclamation-triangle text-warning me-2"></i>Registros Duplicados Encontrados:</h6>
                                <div class="duplicados-list border rounded p-3 bg-light">
                                    <div id="duplicatesList"></div>
                                </div>
                            </div>

                            <div class="text-center mt-4">
                                <button type="button" class="btn btn-primary btn-lg me-3" onclick="location.reload()">
                                    <i class="fas fa-redo me-2"></i>Nueva Importación
                                </button>
                                <a href="clientes.php" class="btn btn-success btn-lg">
                                    <i class="fas fa-users me-2"></i>Ver Clientes Importados
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.3.0/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>
    <script>
        let currentFile = null;
        let workbook = null;
        let currentSheetData = null;
        let debugLog = [];

        // Función de debug
        function addDebugLog(message, type = 'info') {
            const timestamp = new Date().toLocaleTimeString();
            const logEntry = `[${timestamp}] ${type.toUpperCase()}: ${message}`;
            debugLog.push(logEntry);
            
            const debugPanel = document.getElementById('debugLog');
            if (debugPanel) {
                debugPanel.innerHTML = debugLog.slice(-20).join('\n');
                debugPanel.scrollTop = debugPanel.scrollHeight;
            }
            
            console.log(logEntry);
        }

        // Inicialización
        document.addEventListener('DOMContentLoaded', function() {
            addDebugLog('DOM cargado correctamente');
            
            if (typeof XLSX !== 'undefined') {
                addDebugLog('Librería XLSX cargada correctamente');
            } else {
                addDebugLog('ERROR: Librería XLSX no está disponible', 'error');
                mostrarAlerta('danger', 'Error: No se pudo cargar la librería XLSX');
                return;
            }
            
            setupEventListeners();
        });

        function setupEventListeners() {
            addDebugLog('Configurando event listeners...');
            
            // File input
            const fileInput = document.getElementById('fileInput');
            if (fileInput) {
                fileInput.addEventListener('change', handleFileSelect);
                addDebugLog('Event listener del input file configurado');
            }
            
            // Botón seleccionar archivo
            const selectFileBtn = document.getElementById('selectFileBtn');
            if (selectFileBtn) {
                selectFileBtn.addEventListener('click', () => fileInput.click());
                addDebugLog('Event listener del botón configurado');
            }
            
            // Upload area
            const uploadArea = document.getElementById('uploadArea');
            if (uploadArea) {
                uploadArea.addEventListener('click', (e) => {
                    if (!e.target.closest('button')) {
                        fileInput.click();
                    }
                });
                
                // Drag and drop
                uploadArea.addEventListener('dragover', (e) => {
                    e.preventDefault();
                    uploadArea.classList.add('dragover');
                });
                
                uploadArea.addEventListener('dragleave', () => {
                    uploadArea.classList.remove('dragover');
                });
                
                uploadArea.addEventListener('drop', (e) => {
                    e.preventDefault();
                    uploadArea.classList.remove('dragover');
                    const files = e.dataTransfer.files;
                    if (files.length > 0) {
                        handleFile(files[0]);
                    }
                });
            }

            // Otros event listeners
            const sheetSelect = document.getElementById('sheetSelect');
            if (sheetSelect) {
                sheetSelect.addEventListener('change', function() {
                    const previewBtn = document.getElementById('previewBtn');
                    if (previewBtn) {
                        previewBtn.disabled = !this.value;
                    }
                });
            }

            const previewBtn = document.getElementById('previewBtn');
            if (previewBtn) {
                previewBtn.addEventListener('click', showPreview);
            }

            const importBtn = document.getElementById('importBtn');
            if (importBtn) {
                importBtn.addEventListener('click', startImport);
            }
        }

        function handleFileSelect(event) {
            addDebugLog('handleFileSelect ejecutado');
            const files = event.target.files;
            if (files.length > 0) {
                handleFile(files[0]);
            }
        }

        function handleFile(file) {
            addDebugLog(`Procesando archivo: ${file.name}`);
            
            // Validaciones
            const validExtensions = ['.xlsx', '.xls'];
            const fileExtension = file.name.toLowerCase().substring(file.name.lastIndexOf('.'));
            
            if (!validExtensions.includes(fileExtension)) {
                mostrarAlerta('danger', 'Por favor selecciona un archivo Excel válido (.xlsx o .xls)');
                return;
            }

            if (file.size > 50 * 1024 * 1024) {
                mostrarAlerta('danger', 'El archivo es demasiado grande. Máximo permitido: 50MB');
                return;
            }

            currentFile = file;
            
            // Mostrar info del archivo
            document.getElementById('fileName').textContent = file.name;
            document.getElementById('fileSize').textContent = formatFileSize(file.size);
            document.getElementById('fileStatus').textContent = 'Analizando...';
            document.getElementById('fileStatus').className = 'badge bg-warning';
            document.getElementById('fileInfo').style.display = 'block';
            
            // Leer archivo
            const reader = new FileReader();
            reader.onload = function(e) {
                try {
                    workbook = XLSX.read(e.target.result, { type: 'binary' });
                    processWorkbook();
                } catch (error) {
                    addDebugLog(`Error al parsear archivo: ${error.message}`, 'error');
                    mostrarAlerta('danger', 'Error al leer el archivo: ' + error.message);
                    document.getElementById('fileStatus').textContent = 'Error';
                    document.getElementById('fileStatus').className = 'badge bg-danger';
                }
            };
            reader.readAsBinaryString(file);
            updateStep(2);
        }

        function processWorkbook() {
            const sheetNames = workbook.SheetNames;
            addDebugLog(`Hojas encontradas: ${sheetNames.length}`);
            
            document.getElementById('sheetsCount').textContent = sheetNames.length;
            document.getElementById('fileStatus').textContent = 'Procesado';
            document.getElementById('fileStatus').className = 'badge bg-success';
            
            if (sheetNames.length === 1) {
                // Una sola hoja
                const sheetSelect = document.getElementById('sheetSelect');
                sheetSelect.innerHTML = '<option value="">Seleccione una hoja...</option>';
                
                sheetNames.forEach(name => {
                    const option = document.createElement('option');
                    option.value = name;
                    option.textContent = name;
                    sheetSelect.appendChild(option);
                });
                
                sheetSelect.value = sheetNames[0];
                document.getElementById('previewBtn').disabled = false;
                document.getElementById('sheetSelector').style.display = 'block';
            } else {
                // Múltiples hojas
                procesarTodasLasHojas();
            }
        }

        function procesarTodasLasHojas() {
            let todosLosDatos = [];
            let headersPrincipales = [];
            
            workbook.SheetNames.forEach((sheetName, index) => {
                const worksheet = workbook.Sheets[sheetName];
                const jsonData = XLSX.utils.sheet_to_json(worksheet, { header: 1 });
                
                if (jsonData.length > 0) {
                    const headers = jsonData[0] || [];
                    const dataRows = jsonData.slice(1);
                    
                    if (index === 0) {
                        headersPrincipales = headers;
                    }
                    
                    todosLosDatos = todosLosDatos.concat(dataRows);
                }
            });
            
            currentSheetData = [headersPrincipales, ...todosLosDatos];
            
            // Mostrar info de múltiples hojas
            document.getElementById('multiSheetInfo').style.display = 'block';
            document.getElementById('totalRecordsCount').textContent = todosLosDatos.length.toLocaleString();
            
            // Configurar mapeo y mostrar preview
            setupColumnMapping(headersPrincipales);
            displayPreview(headersPrincipales, todosLosDatos.slice(0, 10));
            document.getElementById('previewContainer').style.display = 'block';
            
            mostrarAlerta('success', `Se procesarán ${workbook.SheetNames.length} hojas con ${todosLosDatos.length.toLocaleString()} registros`);
            updateStep(3);
        }

        function showPreview() {
            const selectedSheet = document.getElementById('sheetSelect').value;
            const startRow = parseInt(document.getElementById('startRow').value) - 1;
            
            if (!selectedSheet) {
                mostrarAlerta('warning', 'Por favor selecciona una hoja');
                return;
            }

            const worksheet = workbook.Sheets[selectedSheet];
            const jsonData = XLSX.utils.sheet_to_json(worksheet, { header: 1 });
            
            if (jsonData.length === 0) {
                mostrarAlerta('warning', 'La hoja seleccionada está vacía');
                return;
            }

            currentSheetData = jsonData;
            
            const headers = jsonData[0] || [];
            const dataRows = jsonData.slice(startRow);
            
            setupColumnMapping(headers);
            displayPreview(headers, dataRows.slice(0, 10));
            
            document.getElementById('previewContainer').style.display = 'block';
            updateStep(3);
        }

        function setupColumnMapping(headers) {
            addDebugLog(`Configurando mapeo de columnas con ${headers.length} headers`);
            
            const mappers = ['mapNombre', 'mapTelefono', 'mapCiudad', 'mapOrigen', 'mapFecha'];
            
            mappers.forEach(mapperId => {
                const select = document.getElementById(mapperId);
                if (select) {
                    select.innerHTML = '<option value="">No mapear</option>';
                    
                    headers.forEach((header, index) => {
                        const option = document.createElement('option');
                        option.value = index;
                        option.textContent = header || `Columna ${index + 1}`;
                        select.appendChild(option);
                    });
                    
                    // Agregar event listener para actualizar automáticamente
                    select.addEventListener('change', function() {
                        addDebugLog(`Mapeo cambiado: ${mapperId} = ${this.value}`);
                        setTimeout(() => {
                            mostrarEjemploLimpieza();
                            // Re-renderizar la tabla con los nuevos colores
                            if (currentSheetData && currentSheetData.length > 1) {
                                const headers = currentSheetData[0];
                                const dataRows = currentSheetData.slice(1, 11);
                                displayPreview(headers, dataRows);
                            }
                        }, 100);
                    });
                }
            });
            
            autoMapColumns(headers);
            mostrarEjemploLimpieza();
        }

        function autoMapColumns(headers) {
            const mappings = {
                'mapNombre': ['nombre', 'name', 'cliente', 'razon social'],
                'mapTelefono': ['telefono', 'teléfono', 'phone', 'celular', 'tel'],
                'mapCiudad': ['ciudad', 'city', 'localidad'],
                'mapOrigen': ['origen', 'campaña', 'campaign', 'fuente'],
                'mapFecha': ['fecha', 'date', 'created', 'registro']
            };
            
            Object.keys(mappings).forEach(mapId => {
                const keywords = mappings[mapId];
                const headerIndex = headers.findIndex(header => {
                    if (!header) return false;
                    return keywords.some(keyword => 
                        header.toLowerCase().includes(keyword.toLowerCase())
                    );
                });
                
                if (headerIndex !== -1) {
                    const selectElement = document.getElementById(mapId);
                    if (selectElement) {
                        selectElement.value = headerIndex;
                    }
                }
            });
        }

        function mostrarEjemploLimpieza() {
            addDebugLog('Mostrando ejemplo de limpieza...');
            
            if (currentSheetData && currentSheetData.length > 1) {
                const ejemploData = currentSheetData[1];
                const mapping = {
                    nombre: document.getElementById('mapNombre')?.value || '',
                    telefono: document.getElementById('mapTelefono')?.value || ''
                };
                
                addDebugLog(`Mapping actual: nombre=${mapping.nombre}, telefono=${mapping.telefono}`);
                addDebugLog(`Datos ejemplo: ${JSON.stringify(ejemploData.slice(0, 5))}`);
                
                if (ejemploData && (mapping.nombre !== '' || mapping.telefono !== '')) {
                    const before = [];
                    const after = [];
                    
                    if (mapping.nombre !== '' && ejemploData[mapping.nombre] !== undefined && ejemploData[mapping.nombre] !== null) {
                        const valorOriginal = ejemploData[mapping.nombre];
                        const valorLimpio = limpiarNombre(valorOriginal);
                        before.push(`Nombre: "${valorOriginal}"`);
                        after.push(`Nombre: "${valorLimpio}"`);
                        addDebugLog(`Limpieza nombre: "${valorOriginal}" -> "${valorLimpio}"`);
                    }
                    
                    if (mapping.telefono !== '' && ejemploData[mapping.telefono] !== undefined && ejemploData[mapping.telefono] !== null) {
                        const valorOriginal = ejemploData[mapping.telefono];
                        const valorLimpio = limpiarTelefono(valorOriginal);
                        before.push(`Teléfono: "${valorOriginal}"`);
                        after.push(`Teléfono: "${valorLimpio}"`);
                        addDebugLog(`Limpieza teléfono: "${valorOriginal}" -> "${valorLimpio}"`);
                    }
                    
                    if (before.length > 0) {
                        const beforeElement = document.getElementById('dataBefore');
                        const afterElement = document.getElementById('dataAfter');
                        const previewElement = document.getElementById('cleanedDataPreview');
                        
                        if (beforeElement && afterElement && previewElement) {
                            beforeElement.innerHTML = before.join('<br>');
                            afterElement.innerHTML = after.join('<br>');
                            previewElement.style.display = 'block';
                            addDebugLog('Ejemplo de limpieza actualizado');
                        }
                    } else {
                        addDebugLog('No hay datos para mostrar ejemplo de limpieza');
                    }
                } else {
                    addDebugLog('No hay mapeo configurado o datos de ejemplo');
                }
            } else {
                addDebugLog('No hay datos en currentSheetData para mostrar ejemplo');
            }
        }

        function actualizarVistaPrevia() {
            addDebugLog('Actualizando vista previa manualmente...');
            
            // Actualizar ejemplo de limpieza
            mostrarEjemploLimpieza();
            
            // Re-renderizar la tabla con los nuevos colores
            if (currentSheetData && currentSheetData.length > 1) {
                const headers = currentSheetData[0];
                const dataRows = currentSheetData.slice(1, 11); // Primeras 10 filas
                displayPreview(headers, dataRows);
                addDebugLog('Vista previa actualizada manualmente');
            } else {
                addDebugLog('No hay datos para actualizar la vista previa', 'warning');
            }
        }

        function displayPreview(headers, dataRows) {
            addDebugLog(`Mostrando preview con ${headers.length} headers y ${dataRows.length} filas`);
            
            const previewHeaders = document.getElementById('previewHeaders');
            const previewBody = document.getElementById('previewBody');
            
            if (!previewHeaders || !previewBody) {
                addDebugLog('ERROR: Elementos de preview no encontrados', 'error');
                return;
            }
            
            // Headers
            previewHeaders.innerHTML = headers.map(header => 
                `<th>${header || 'Sin nombre'}</th>`
            ).join('');
            
            // Obtener mapping actual
            const mapping = {
                nombre: document.getElementById('mapNombre')?.value || '',
                telefono: document.getElementById('mapTelefono')?.value || ''
            };
            
            addDebugLog(`Preview con mapping: nombre=${mapping.nombre}, telefono=${mapping.telefono}`);
            
            // Rows con limpieza visual
            previewBody.innerHTML = dataRows.map((row, rowIndex) => {
                if (!row) return '';
                
                return `<tr>${headers.map((_, index) => {
                    let valor = '';
                    if (row[index] !== undefined && row[index] !== null) {
                        valor = String(row[index]);
                    }
                    
                    let claseExtra = '';
                    let valorMostrar = valor;
                    
                    // Aplicar limpieza visual y colores
                    if (index == mapping.nombre && mapping.nombre !== '' && valor) {
                        valorMostrar = limpiarNombre(valor);
                        claseExtra = 'table-success';
                    } else if (index == mapping.telefono && mapping.telefono !== '' && valor) {
                        valorMostrar = limpiarTelefono(valor);
                        claseExtra = 'table-info';
                    }
                    
                    // Limitar longitud para mostrar
                    if (valorMostrar.length > 50) {
                        valorMostrar = valorMostrar.substring(0, 47) + '...';
                    }
                    
                    return `<td class="${claseExtra}" title="${valor}">${valorMostrar}</td>`;
                }).join('')}</tr>`;
            }).join('');
            
            addDebugLog('Preview actualizado correctamente');
        }

        function startImport() {
            addDebugLog('Iniciando importación...');
            
            const mapping = {
                nombre: document.getElementById('mapNombre')?.value || '',
                telefono: document.getElementById('mapTelefono')?.value || '',
                ciudad: document.getElementById('mapCiudad')?.value || '',
                origen: document.getElementById('mapOrigen')?.value || '',
                fecha: document.getElementById('mapFecha')?.value || ''
            };
            
            if (!mapping.telefono) {
                mostrarAlerta('danger', 'El campo TELÉFONO es obligatorio para mapear');
                return;
            }
            
            const startRow = document.getElementById('startRow') ? 
                (parseInt(document.getElementById('startRow').value) - 1) : 1;
            const dataRows = currentSheetData.slice(startRow);
            
            document.getElementById('previewContainer').style.display = 'none';
            document.getElementById('progressContainer').style.display = 'block';
            updateStep(4);
            
            // Inicializar contadores
            document.getElementById('errorCount').textContent = '0';
            
            processBatch(dataRows, mapping, 0, {
                processed: 0,
                success: 0,
                duplicates: 0,
                errors: 0,
                duplicatesList: []
            });
        }

        function processBatch(allRows, mapping, startIndex, stats) {
            const batchSize = 50;
            const endIndex = Math.min(startIndex + batchSize, allRows.length);
            const batch = allRows.slice(startIndex, endIndex);
            
            if (batch.length === 0) {
                showResults(stats);
                return;
            }
            
            // Preparar datos del lote con limpieza
            const batchData = batch.map(row => {
                const cliente = {};
                Object.keys(mapping).forEach(field => {
                    const colIndex = mapping[field];
                    if (colIndex !== '' && colIndex !== null) {
                        let valor = row[colIndex] || '';
                        
                        if (field === 'nombre') {
                            valor = limpiarNombre(valor);
                        } else if (field === 'telefono') {
                            valor = limpiarTelefono(valor);
                        } else if (field === 'ciudad') {
                            valor = limpiarTexto(valor);
                        }
                        
                        cliente[field] = valor;
                    }
                });
                return cliente;
            }).filter(cliente => cliente.telefono && cliente.telefono.length >= 10);
            
            // Actualizar mensaje de estado
            document.getElementById('statusMessage').innerHTML = 
                `<i class="fas fa-cogs fa-spin me-2"></i>Procesando lote ${Math.floor(startIndex/batchSize) + 1}... Limpiando y validando ${batchData.length} registros`;
            
            // Enviar lote via AJAX
            const formData = new FormData();
            formData.append('action', 'importar_clientes_lote');
            formData.append('clientes', JSON.stringify(batchData));
            // Nota: Reemplazar con el token CSRF real
            formData.append('csrf_token', CSRF_TOKEN);
            
            fetch('ajax_clientes.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    stats.processed += data.stats.processed;
                    stats.success += data.stats.success;
                    stats.duplicates += data.stats.duplicates;
                    stats.errors += data.stats.errors;
                    stats.duplicatesList.push(...data.stats.duplicatesList);
                    
                    const progress = Math.round((endIndex / allRows.length) * 100);
                    updateProgress(progress, stats);
                    
                    setTimeout(() => {
                        processBatch(allRows, mapping, endIndex, stats);
                    }, 100);
                } else {
                    mostrarAlerta('danger', 'Error en la importación: ' + data.message);
                    stats.errors += batchData.length;
                    updateProgress(100, stats);
                    showResults(stats);
                }
            })
            .catch(error => {
                mostrarAlerta('danger', 'Error de conexión: ' + error.message);
                stats.errors += batchData.length;
                updateProgress(100, stats);
                showResults(stats);
            });
        }

        // Funciones de limpieza simplificadas para preview
        function limpiarNombre(nombre) {
            if (!nombre || typeof nombre !== 'string' || nombre.trim() === '') {
                return 'Sin Nombre';
            }
            
            nombre = nombre.trim();
            nombre = nombre.replace(/\w\S*/g, (txt) => 
                txt.charAt(0).toUpperCase() + txt.substr(1).toLowerCase()
            );
            
            return nombre || 'Sin Nombre';
        }

        function limpiarTelefono(telefono) {
            if (!telefono || typeof telefono !== 'string') {
                return '';
            }
            
            telefono = telefono.replace(/[^0-9]/g, '');
            
            if (!telefono) return '';
            
            if (telefono.startsWith('58') && telefono.length > 10) {
                telefono = telefono.substring(2);
            }
            
            if (telefono.startsWith('0')) {
                telefono = telefono.substring(1);
            }
            
            if (telefono.length === 10 && telefono.startsWith('4')) {
                return '58' + telefono;
            }
            
            if (telefono.length === 7) {
                return '58414' + telefono;
            }
            
            return '58' + telefono;
        }

        function limpiarTexto(texto) {
            if (!texto || typeof texto !== 'string') {
                return '';
            }
            return texto.trim();
        }

        function updateProgress(percentage, stats) {
            const progressBar = document.getElementById('progressBar');
            const progressText = document.getElementById('progressText');
            
            progressBar.style.width = percentage + '%';
            progressText.textContent = percentage + '%';
            
            document.getElementById('processedCount').textContent = stats.processed.toLocaleString();
            document.getElementById('successCount').textContent = stats.success.toLocaleString();
            document.getElementById('duplicateCount').textContent = stats.duplicates.toLocaleString();
            document.getElementById('errorCount').textContent = stats.errors.toLocaleString();
            
            if (percentage === 100) {
                document.getElementById('statusMessage').innerHTML = 
                    `<i class="fas fa-check text-success me-2"></i>Importación completada. Procesados: ${stats.processed.toLocaleString()}, Exitosos: ${stats.success.toLocaleString()}`;
                document.getElementById('statusMessage').className = 'alert alert-success';
            }
        }

        function showResults(stats) {
            document.getElementById('progressContainer').style.display = 'none';
            document.getElementById('resultsContainer').style.display = 'block';
            
            document.getElementById('finalProcessed').textContent = stats.processed.toLocaleString();
            document.getElementById('finalSuccess').textContent = stats.success.toLocaleString();
            document.getElementById('finalDuplicates').textContent = stats.duplicates.toLocaleString();
            document.getElementById('finalErrors').textContent = stats.errors.toLocaleString();
            
            if (stats.duplicatesList && stats.duplicatesList.length > 0) {
                document.getElementById('duplicatesSection').style.display = 'block';
                document.getElementById('duplicatesList').innerHTML = stats.duplicatesList.map(dup => 
                    `<div class="mb-2 p-2 bg-white rounded border">
                        <strong>${dup.nombre || 'Sin nombre'}</strong> - ${dup.telefono}
                        <small class="text-muted d-block">Ya existe en la base de datos</small>
                    </div>`
                ).join('');
            }
            
            updateStep(5);
        }

        function updateStep(stepNumber) {
            document.querySelectorAll('.step').forEach(step => {
                step.classList.remove('active');
            });
            
            for (let i = 1; i < stepNumber; i++) {
                document.getElementById(`step${i}`).classList.add('completed');
            }
            
            if (stepNumber <= 5) {
                document.getElementById(`step${stepNumber}`).classList.add('active');
            }
        }

        function formatFileSize(bytes) {
            if (bytes === 0) return '0 Bytes';
            const k = 1024;
            const sizes = ['Bytes', 'KB', 'MB', 'GB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
        }

        function mostrarAlerta(tipo, mensaje) {
            const alertContainer = document.createElement('div');
            alertContainer.className = `alert alert-${tipo} alert-dismissible fade show position-fixed`;
            alertContainer.style.cssText = 'top: 20px; right: 20px; z-index: 1055; min-width: 300px;';
            alertContainer.innerHTML = `
                ${mensaje}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            `;
            document.body.appendChild(alertContainer);
            
            setTimeout(() => {
                if (alertContainer.parentNode) {
                    alertContainer.remove();
                }
            }, 5000);
        }

        addDebugLog('Script cargado, esperando DOM...');
    </script>
</body>
</html>