<?php
require_once 'includes/seguridad.php';

$jornada_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
if (!$jornada_id) {
    header('Location: mis-rutas.php');
    exit();
}

$stmt = $conn->prepare("SELECT j.id, j.ciudad_nombre, j.fecha_jornada, r.coordinador_id, r.id as ruta_id FROM jornadas j JOIN rutas r ON j.ruta_id = r.id WHERE j.id = ?");
$stmt->execute([$jornada_id]);
$jornada = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$jornada) {
    header("Location: mis-rutas.php?error=jornada_no_encontrada&jornada_id={$jornada_id}");
    exit();
}

// --- LÓGICA DE ACCESO Y FECHA ---
$es_superadmin = puede('gestionar_jornadas_superadmin');

// 1. Verificación de Permiso: Denegar si no es superadmin NI el coordinador asignado.
if (!$es_superadmin && $jornada['coordinador_id'] != $_SESSION['user_id']) {
    registrar_accion('intento_acceso_no_autorizado', "Intento de acceder a jornada ajena. Jornada ID: {$jornada_id}");
    header("Location: mis-rutas.php?error=acceso_denegado&jornada_id={$jornada_id}");
    exit();
}

// 2. Verificación de Fecha: SOLO se ejecuta si el usuario NO es superadmin.
if (!$es_superadmin) {
    $hoy = date('Y-m-d');
    if ($jornada['fecha_jornada'] != $hoy) {
        registrar_accion('acceso_fuera_fecha', "Intento de acceder a jornada fuera de fecha. Jornada ID: {$jornada_id}");
        header("Location: mis-rutas.php?error=fecha_incorrecta&jornada_id={$jornada_id}");
        exit();
    }
}

// (Opcional) Registrar acceso de superadmin a jornada ajena para auditoría
if ($es_superadmin && $jornada['coordinador_id'] != $_SESSION['user_id']) {
    registrar_accion('acceso_superadmin_jornada_ajena', "Superadmin ID {$_SESSION['user_id']} accedió a jornada ID {$jornada_id} (coordinador: {$jornada['coordinador_id']})");
}

$page_title = 'Gestionando Jornada: ' . htmlspecialchars($jornada['ciudad_nombre']);
include_once 'templates/header.php';
?>

<style>
.paciente-row { background: rgba(0,0,0,0.2); padding: 1rem; border-radius: 8px; margin-bottom: 1rem; cursor: pointer; transition: background-color 0.2s ease-in-out; }
.paciente-row:hover { background-color: rgba(var(--primary-color-rgb), 0.2); }
.badge-estado { padding: 0.4em 0.7em; border-radius: 10px; font-weight: 500; text-transform: capitalize; min-width: 110px; text-align: center; }
.paciente-numero { font-size: 1.5rem; font-weight: bold; color: var(--primary-color); }
.suministro-item, .medicamento-item, .examen-item { display: grid; gap: 1rem; align-items: center; margin-bottom: 0.75rem; padding: 0.75rem; background: rgba(0,0,0,0.2); border-radius: 6px; border: 1px solid rgba(255,255,255,0.1); }
.suministro-item { grid-template-columns: 1fr 100px 80px; }
.medicamento-item { grid-template-columns: 1fr 80px 120px 50px; }
.examen-item { grid-template-columns: 1fr 120px 50px; }
#modalGestionarPaciente .form-control { background-color: rgba(15, 23, 42, 0.7); border: 1px solid #475569; color: var(--light-text); }
.suministro-item .form-control, .medicamento-item .form-control, .examen-item .form-control { text-align: center; }
.suministro-item label, .medicamento-item label, .examen-item label { font-weight: 500; }
.input-group-text-custom { position: absolute; left: 15px; top: 50%; transform: translateY(-50%); color: var(--light-text); z-index: 5; }
.input-con-simbolo input { padding-left: 50px !important; }
.seccion-titulo { border-bottom: 2px solid var(--primary-color); padding-bottom: 0.5rem; margin-bottom: 1.5rem; }
</style>

<div class="main-header">
    <div>
        <a href="mis-rutas.php" class="btn btn-outline-light mb-2"><i class="fa-solid fa-arrow-left"></i> Volver a Mis Rutas</a>
        <h1><?php echo $page_title; ?></h1>
    </div>
</div>

<div class="info-card mb-4">
    <input type="text" id="searchInput" class="form-control form-control-dark" placeholder="Buscar paciente por nombre o cédula...">
</div>

<div id="pacientes-container"></div>

<div class="modal fade ami-modal-theme" id="modalGestionarPaciente" tabindex="-1">
  <div class="modal-dialog" style="max-width: 95%; width: 1400px;">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title">Gestionar Paciente: <span id="nombrePacienteModal"></span></h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body">
        <form id="formGestionarPaciente" enctype="multipart/form-data">
          <input type="hidden" name="action" value="actualizar_paciente_jornada">
          <input type="hidden" id="jornada_id_hidden" name="jornada_id" value="<?php echo $jornada_id; ?>">
          <input type="hidden" id="jornada_cliente_id" name="jornada_cliente_id">
          
          <div class="row">
            <div class="col-lg-7">
                <div class="mb-4">
                    <h6 class="text-primary mb-3 seccion-titulo"><i class="fa-solid fa-user-cog me-2"></i>Datos Generales y Pago</h6>
                    <div class="row">
                        <div class="col-md-12 mb-3"><label class="ami-form-label">Estado de la Cita</label><select name="estado_cita" class="form-select" required><option value="Agendado">Agendado</option><option value="En consultorio">En consultorio</option><option value="Atendido">Atendido</option><option value="No asistió">No asistió</option><option value="Cancelado">Cancelado</option></select></div>
                    </div>
                    
                    <div id="pagos-detallados-container" class="row">
                        <div class="col-md-4 mb-3"><label class="ami-form-label">Monto en Divisas</label><div class="input-group input-con-simbolo"><span class="input-group-text-custom">$</span><input type="number" name="pago_divisas" class="form-control" min="0" step="0.01" placeholder="0.00"></div></div>
                        <div class="col-md-4 mb-3"><label class="ami-form-label">Monto en Zelle</label><div class="input-group input-con-simbolo"><span class="input-group-text-custom">$</span><input type="number" name="pago_zelle" class="form-control" min="0" step="0.01" placeholder="0.00"></div></div>
                        <div class="col-md-4 mb-3"><label class="ami-form-label">Monto en Pesos</label><div class="input-group input-con-simbolo"><span class="input-group-text-custom">Cop</span><input type="number" name="pago_pesos" class="form-control" min="0" step="0.01" placeholder="0.00"></div></div>
                        <div class="col-md-4 mb-3"><label class="ami-form-label">Monto en Efectivo (Bs)</label><div class="input-group input-con-simbolo"><span class="input-group-text-custom">Bs</span><input type="number" name="pago_bs_efectivo" class="form-control" min="0" step="0.01" placeholder="0.00"></div></div>
                        <div class="col-md-4 mb-3"><label class="ami-form-label">Monto en Punto</label><div class="input-group input-con-simbolo"><span class="input-group-text-custom">Bs</span><input type="number" name="pago_punto" class="form-control" min="0" step="0.01" placeholder="0.00"></div></div>
                        <div class="col-md-4 mb-3"><label class="ami-form-label">Monto en Pago Móvil</label><div class="input-group input-con-simbolo"><span class="input-group-text-custom">Bs</span><input type="number" name="pago_movil" class="form-control" min="0" step="0.01" placeholder="0.00"></div></div>
                    </div>

                    <div class="mb-3"><label class="ami-form-label">Subir Comprobante (Opcional)</label><input type="file" name="comprobante_pago" class="form-control" accept="image/*,.pdf"><div id="comprobante-actual"></div></div>
                    <div class="mb-3"><label class="ami-form-label">Notas Adicionales</label><textarea name="notas_coordinador" class="form-control" rows="3" placeholder="Observaciones de la consulta..."></textarea></div>
                </div>

                <div class="mb-4"><h6 class="text-success mb-3 seccion-titulo"><i class="fa-solid fa-stethoscope me-2"></i>Exámenes Realizados</h6><div id="examenes-realizados-container" class="mb-3"></div><button type="button" class="btn btn-sm btn-outline-success" id="btn-add-examen"><i class="fa-solid fa-plus"></i> Añadir Examen</button></div>
                <div class="mb-4"><h6 class="text-info mb-3 seccion-titulo"><i class="fa-solid fa-pills me-2"></i>Venta de Medicamentos</h6><div id="medicamentos-vendidos-container" class="mb-3"></div><button type="button" class="btn btn-sm btn-outline-info" id="btn-add-medicamento"><i class="fa-solid fa-plus"></i> Añadir Medicamento para Venta</button></div>
            </div>
            <div class="col-lg-5">
                <div class="mb-4"><h6 class="text-warning mb-3 seccion-titulo"><i class="fa-solid fa-box me-2"></i>Suministros de Operación (Consumo)</h6><p class="text-white-50 small mb-3">Materiales usados en la consulta que se descuentan del stock.</p><div id="suministros-container"></div></div>
            </div>
          </div>
        </form>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
        <button type="button" class="btn btn-primary" id="btnGuardarGestion">Guardar Cambios</button>
      </div>
    </div>
  </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const jornadaId = <?php echo $jornada_id; ?>;
    const rutaId = <?php echo $jornada['ruta_id']; ?>;
    const modalGestion = new bootstrap.Modal(document.getElementById('modalGestionarPaciente'));
    const csrfToken = '<?php echo $_SESSION['csrf_token']; ?>';
    let pacientesData = [], suministrosRuta = [], medicamentosEnRuta = [];

    function getEstadoBadge(estado) {
        const colores = { 'Agendado': 'bg-secondary', 'En consultorio': 'bg-info text-dark', 'Atendido': 'bg-success', 'No asistió': 'bg-warning text-dark', 'Cancelado': 'bg-danger' };
        return `<span class="badge-estado ${colores[estado] || 'bg-light text-dark'}">${estado || 'Indefinido'}</span>`;
    }

    function renderizarPacientes(pacientes) {
        const container = document.getElementById('pacientes-container');
        container.innerHTML = '';
        if (!pacientes || pacientes.length === 0) {
            container.innerHTML = '<div class="info-card">No se encontraron pacientes para el filtro actual.</div>';
            return;
        }
        pacientes.forEach((p, index) => {
            const comprobanteHtml = p.comprobante_pago ? `<a href="/uploads/comprobantes/${p.comprobante_pago}" target="_blank" class="btn btn-sm btn-outline-info" data-no-modal="true">Ver Comp.</a>` : '';
            let iconoSexo = p.sexo === 'M' ? '<i class="fa-solid fa-mars text-info" title="Masculino"></i>' : (p.sexo === 'F' ? '<i class="fa-solid fa-venus text-danger" title="Femenino"></i>' : '');
            const infoPersonalHtml = `C.I: ${p.cedula} | ${p.edad || '?'} años ${iconoSexo}`;
            const pacienteHtml = `<div class="info-card paciente-row" data-paciente-id="${p.jornada_cliente_id}"><div class="row align-items-center"><div class="col-1 text-center"><span class="paciente-numero">${index + 1}</span></div><div class="col-4"><h5>${p.nombre}</h5><small>${infoPersonalHtml} | Tel: ${p.telefono || 'N/A'}</small></div><div class="col-2 text-center">${getEstadoBadge(p.estado_cita)}</div><div class="col-4"><small><b>Total Facturado:</b> <span class="font-monospace">$${parseFloat(p.total_facturado || 0).toFixed(2)}</span></small></div><div class="col-1 text-end">${comprobanteHtml}</div></div></div>`;
            container.innerHTML += pacienteHtml;
        });
    }

    async function cargarDatosIniciales() {
        try {
            const [pacientesRes, stockRes] = await Promise.all([
                fetch(`/api/coordinado_ruta_ajax.php?action=get_jornada_coordinador&id=${jornadaId}`),
                fetch(`/api/coordinado_ruta_ajax.php?action=get_stock_ruta&ruta_id=${rutaId}`)
            ]);
            const pacientesResponse = await pacientesRes.json();
            if (!pacientesResponse.success) throw new Error(pacientesResponse.message);
            pacientesData = pacientesResponse.data.pacientes;
            const stockResponse = await stockRes.json();
            if (!stockResponse.success) throw new Error(stockResponse.message);
            suministrosRuta = stockResponse.data.filter(item => item.categoria !== 'Medicamentos');
            medicamentosEnRuta = stockResponse.data.filter(item => item.categoria === 'Medicamentos');
            renderizarPacientes(pacientesData);
        } catch (error) {
            Swal.fire('Error', `No se pudieron cargar los datos de la jornada: ${error.message}`, 'error');
        }
    }
    
    document.getElementById('searchInput').addEventListener('input', function() {
        const filtro = this.value.toLowerCase();
        const pacientesFiltrados = pacientesData.filter(p => p.nombre.toLowerCase().includes(filtro) || p.cedula.includes(filtro));
        renderizarPacientes(pacientesFiltrados);
    });

    document.getElementById('pacientes-container').addEventListener('click', function(event) {
        if (event.target.closest('[data-no-modal="true"]')) return;
        const fila = event.target.closest('.paciente-row');
        if (fila) abrirModalGestion(fila.dataset.pacienteId);
    });

    function abrirModalGestion(jornadaClienteId) {
        const paciente = pacientesData.find(p => p.jornada_cliente_id == jornadaClienteId);
        if (!paciente) return;
        
        const form = document.getElementById('formGestionarPaciente');
        form.reset();
        document.getElementById('nombrePacienteModal').textContent = paciente.nombre;
        document.getElementById('jornada_cliente_id').value = paciente.jornada_cliente_id;
        form.querySelector('[name="estado_cita"]').value = paciente.estado_cita || 'Agendado';
        
        ['pago_divisas', 'pago_zelle', 'pago_bs_efectivo', 'pago_punto', 'pago_movil', 'pago_pesos'].forEach(campo => {
            form.querySelector(`[name="${campo}"]`).value = parseFloat(paciente[campo]) > 0 ? paciente[campo] : '';
        });
        form.querySelector('[name="notas_coordinador"]').value = paciente.notas_coordinador || '';

        const suministrosContainer = document.getElementById('suministros-container');
        suministrosContainer.innerHTML = '';
        suministrosRuta.forEach(item => {
            const consumido = (paciente.consumo && Array.isArray(paciente.consumo)) ? paciente.consumo.find(c => c.producto_id == item.producto_id) : null;
            const cantidadConsumida = consumido ? consumido.cantidad_usada : 0;
            const stockActual = item.cantidad + cantidadConsumida;
            suministrosContainer.innerHTML += `<div class="suministro-item" data-sku="${item.sku}"><label for="suministro-${item.producto_id}">${item.nombre}</label><span class="text-white-50">Stock: ${item.cantidad}</span><input type="number" name="suministros[${item.producto_id}]" id="suministro-${item.producto_id}" class="form-control" value="${cantidadConsumida}" min="0" max="${stockActual}"></div>`;
        });
        aplicarLogicaGenero(paciente.sexo);

        document.getElementById('examenes-realizados-container').innerHTML = '';
        if (paciente.examenes && paciente.examenes.length > 0) paciente.examenes.forEach(examen => agregarExamenRealizado(examen));
        document.getElementById('medicamentos-vendidos-container').innerHTML = '';
        if (paciente.medicamentos && paciente.medicamentos.length > 0) paciente.medicamentos.forEach(med => agregarMedicamentoVendido(med));
        document.getElementById('comprobante-actual').innerHTML = paciente.comprobante_pago ? `<small class="d-block mt-2">Archivo actual: <a href="/uploads/comprobantes/${paciente.comprobante_pago}" target="_blank" data-no-modal="true">${paciente.comprobante_pago}</a></small>` : '';

        modalGestion.show();
    }

    function aplicarLogicaGenero(sexo) {
        const psaItem = document.querySelector('.suministro-item[data-sku="PR-PSA"]'), hpItem = document.querySelector('.suministro-item[data-sku="PR-HP"]');
        if (psaItem) psaItem.style.display = (sexo === 'F') ? 'none' : 'grid';
        if (sexo === 'M' && psaItem && hpItem) {
            const psaInput = psaItem.querySelector('input'), hpInput = hpItem.querySelector('input');
            const handler = (e) => { if (parseInt(e.target.value, 10) > 0) (e.target === psaInput ? hpInput : psaInput).value = 0; };
            psaInput.addEventListener('input', handler);
            hpInput.addEventListener('input', handler);
        }
    }

    function agregarExamenRealizado(examen = {}) {
        const container = document.getElementById('examenes-realizados-container'), index = Date.now();
        const examenesTipos = [{ value: 'Bioescaner', label: 'Bioescaner' }, { value: 'Hemograma', label: 'Hemograma' }, { value: 'Glicemia', label: 'Glicemia' }, { value: 'Colesterol', label: 'Colesterol' }, { value: 'Ecografia', label: 'Ecografía' }, { value: 'Radiografia', label: 'Radiografía' }];
        const tiposOptions = examenesTipos.map(e => `<option value="${e.value}" ${examen.nombre_examen === e.value ? 'selected' : ''}>${e.label}</option>`).join('');
        const itemHtml = `<div class="examen-item" id="examen-${index}"><select name="examenes[${index}][tipo]" class="form-select" required><option value="">Seleccionar Examen</option>${tiposOptions}</select><div class="input-group input-con-simbolo"><span class="input-group-text-custom">$</span><input type="number" name="examenes[${index}][precio]" class="form-control" min="0" step="0.01" value="${examen.precio || ''}" placeholder="Precio" required></div><button type="button" class="btn btn-sm btn-danger" onclick="document.getElementById('examen-${index}').remove()"><i class="fa-solid fa-trash"></i></button></div>`;
        container.insertAdjacentHTML('beforeend', itemHtml);
    }

    function agregarMedicamentoVendido(med = {}) {
        const container = document.getElementById('medicamentos-vendidos-container'), index = Date.now();
        const stockOptions = medicamentosEnRuta.map(s => `<option value="${s.producto_id}" ${med.producto_id == s.producto_id ? 'selected' : ''}>${s.nombre} (Stock: ${s.cantidad})</option>`).join('');
        const itemHtml = `<div class="medicamento-item" id="med-${index}"><select name="medicamentos[${index}][producto_id]" class="form-select"><option value="">Seleccione un medicamento</option>${stockOptions}</select><input type="number" name="medicamentos[${index}][cantidad]" class="form-control" min="1" value="${med.cantidad || 1}" placeholder="Cant."><div class="input-group input-con-simbolo"><span class="input-group-text-custom">$</span><input type="number" name="medicamentos[${index}][precio]" class="form-control" min="0" step="0.01" value="${med.precio || ''}" placeholder="Precio Unit."></div><button type="button" class="btn btn-sm btn-danger" onclick="document.getElementById('med-${index}').remove()"><i class="fa-solid fa-trash"></i></button></div>`;
        container.insertAdjacentHTML('beforeend', itemHtml);
    }
    
    document.getElementById('btn-add-examen').addEventListener('click', () => agregarExamenRealizado());
    document.getElementById('btn-add-medicamento').addEventListener('click', () => agregarMedicamentoVendido());
    
    document.getElementById('btnGuardarGestion').addEventListener('click', function() {
        const formData = new FormData(document.getElementById('formGestionarPaciente'));
        formData.append('csrf_token', csrfToken);
        Swal.fire({ title: 'Guardando...', allowOutsideClick: false, didOpen: () => Swal.showLoading() });
        fetch('/api/coordinado_ruta_ajax.php', { method: 'POST', body: formData })
        .then(res => res.ok ? res.json() : Promise.reject(res))
        .then(data => {
            Swal.close();
            if(data.success) {
                Swal.fire('Éxito', data.message, 'success');
                modalGestion.hide();
                cargarDatosIniciales();
            } else { Swal.fire('Error', data.message, 'error'); }
        })
        .catch(error => {
            Swal.close();
            Swal.fire('Error', 'Error de conexión al guardar los datos', 'error');
        });
    });

    cargarDatosIniciales();
});
</script>

<?php include_once 'templates/footer.php'; ?>