<?php
require_once 'includes/seguridad.php'; 
// Obtenemos el ID del lote desde la URL. Si no existe o no es un número, lo sacamos.
$lote_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
if (!$lote_id) {
    header('Location: prospectos-admin.php');
    exit();
}

// Consultamos la información básica del lote para el título y la cabecera
$stmt_lote_info = $conn->prepare("SELECT l.nombre, u.nombre as operador_nombre FROM lotes l LEFT JOIN usuarios u ON l.operador_id = u.id WHERE l.id = ?");
$stmt_lote_info->execute([$lote_id]);
$lote_info = $stmt_lote_info->fetch(PDO::FETCH_ASSOC);

if (!$lote_info) {
    // Si el lote no existe, lo devolvemos a la página principal.
    header('Location: prospectos-admin.php');
    exit();
}

$page_title = 'Detalle del Lote: ' . htmlspecialchars($lote_info['nombre']);
include_once 'templates/header.php';

if (!puede('gestionar_prospectos_admin')) {
    echo "<main class='ami-content'><div class='info-card'><h1>Acceso Denegado</h1></div></main>";
    include_once 'templates/footer.php';
    exit();
}
?>

<style>
.telefono-ofuscado {
    font-style: italic;
    opacity: 0.7;
    cursor: pointer;
}
.nota-texto {
    white-space: pre-wrap;
    max-height: 120px;
    overflow-y: auto;
    font-size: 0.85em;
    background-color: rgba(0,0,0,0.2);
    padding: 8px;
    border-radius: 5px;
    border: 1px solid var(--border-color);
}
.badge-estado {
    font-size: 0.8em;
    padding: 0.4em 0.7em;
    border-radius: 10px;
}
</style>

<div class="main-header">
    <div>
        <a href="prospectos-admin.php" class="btn btn-outline-light mb-2"><i class="fa-solid fa-arrow-left"></i> Volver a Lotes</a>
        <h1><?php echo $page_title; ?></h1>
        <p class="lead">Asignado a: <strong><?php echo htmlspecialchars($lote_info['operador_nombre'] ?? 'N/A'); ?></strong></p>
    </div>
</div>

<div class="info-card">
    <div class="d-flex justify-content-end mb-3">
        <input type="text" id="filtroProspectos" class="form-control form-control-dark" style="max-width: 400px;" placeholder="Buscar por nombre, teléfono, ciudad o estado...">
    </div>
    <table class="ami-table">
        <thead>
            <tr>
                <th>Nombre</th>
                <th>Teléfono</th>
                <th>Ciudad</th>
                <th>Estado</th>
                <th>Notas del Operador</th>
                <th>Última Llamada</th>
            </tr>
        </thead>
        <tbody id="tablaDetallesLote">
            <tr><td colspan="6" class="text-center">Cargando prospectos...</td></tr>
        </tbody>
    </table>
</div>

<script>
function cargarDetalles() {
    const loteId = <?php echo $lote_id; ?>;
    const tablaBody = document.getElementById('tablaDetallesLote');
    
    fetch(`/api/prospectos_ajax.php?action=get_detalles_lote&id=${loteId}`)
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                tablaBody.innerHTML = '';
                if (data.data.length > 0) {
                    data.data.forEach(p => {
                        const fecha = p.fecha_ultima_llamada ? new Date(p.fecha_ultima_llamada).toLocaleString() : 'N/A';
                        const telefonoHtml = `<span class="telefono-ofuscado" title="Clic para revelar" onclick="revelar(this, '${p.telefono}')">${ofuscarTelefono(p.telefono)}</span>`;
                        const estadoBadge = `<span class="badge badge-estado" style="background-color: ${getColorPorEstado(p.estado)};">${p.estado}</span>`;

                        tablaBody.innerHTML += `
                            <tr>
                                <td>${p.nombre}</td>
                                <td class="font-monospace">${telefonoHtml}</td>
                                <td>${p.ciudad}</td>
                                <td>${estadoBadge}</td>
                                <td><div class="nota-texto">${p.notas || 'Sin notas.'}</div></td>
                                <td>${fecha}</td>
                            </tr>
                        `;
                    });
                } else {
                    tablaBody.innerHTML = '<tr><td colspan="6" class="text-center">Este lote aún no tiene prospectos o ya fueron procesados.</td></tr>';
                }
            } else {
                Swal.fire('Error', data.message, 'error');
                tablaBody.innerHTML = `<tr><td colspan="6" class="text-center text-danger">${data.message}</td></tr>`;
            }
        });
}

function revelar(elemento, telefono) {
    elemento.outerHTML = `<a href="tel:${telefono}">${telefono}</a>`;
}

function ofuscarTelefono(tel) {
    if (tel.length > 8) {
        return tel.substring(0, 4) + 'XXXX' + tel.substring(tel.length - 3);
    }
    return 'XXX-XXXX';
}

function getColorPorEstado(estado) {
    switch(estado) {
        case 'interesado': return 'rgba(25, 135, 84, 0.3)'; // Verde
        case 'contactado': return 'rgba(13, 202, 240, 0.3)'; // Azul claro
        case 'no contesta': return 'rgba(255, 193, 7, 0.3)'; // Amarillo
        case 'descartado': return 'rgba(220, 53, 69, 0.3)'; // Rojo
        default: return 'rgba(108, 117, 125, 0.3)'; // Gris (nuevo)
    }
}

document.addEventListener('DOMContentLoaded', cargarDetalles);

document.getElementById('filtroProspectos').addEventListener('keyup', function() {
    const filtro = this.value.toLowerCase();
    const filas = document.getElementById('tablaDetallesLote').getElementsByTagName('tr');
    for (let fila of filas) {
        fila.style.display = fila.textContent.toLowerCase().includes(filtro) ? '' : 'none';
    }
});
</script>

<?php include_once 'templates/footer.php'; ?>