<?php
require_once 'includes/seguridad.php';

// Validar que se reciba un ID numérico por la URL
$nomina_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
if (!$nomina_id) {
    header('Location: nominas.php?error=id_invalido');
    exit();
}

// Permisos para ver el módulo
if (!puede('ver_nomina') && !puede('gestionar_nomina')) {
    header('Location: inicio.php?error=acceso_denegado');
    exit();
}

$page_title = 'Detalle de Nómina #' . $nomina_id;
include_once 'templates/header.php';
?>

<style>
    .detalle-header {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 1rem;
        margin-bottom: 1.5rem;
    }
    .detalle-info-box {
        background-color: rgba(0,0,0,0.2);
        padding: 1rem;
        border-radius: .375rem;
        border: 1px solid var(--border-color);
    }
    .detalle-info-box .label {
        font-size: 0.9em;
        color: var(--light-text-secondary);
    }
    .detalle-info-box .value {
        font-size: 1.2em;
        font-weight: bold;
        color: var(--light-text);
    }
    .text-success { color: #20c997 !important; }
    .total-final-usd { color: var(--primary-color); }
    
    /* Estilos para filas pagadas */
    .fila-pagada {
        background-color: rgba(40, 167, 69, 0.1) !important;
        transition: background-color 0.3s ease;
    }
    
    .fila-pagada td {
        border-color: rgba(40, 167, 69, 0.3) !important;
    }
    
    .checkbox-empleado {
        transform: scale(1.2);
        margin-right: 0.5rem;
    }
</style>

<div class="main-header">
    <a href="nominas.php" class="btn-back"><i class="fa-solid fa-arrow-left"></i> Volver</a>
    <h1 id="main-title">Detalle de Nómina</h1>
</div>

<div class="info-card" id="nomina-detalle-container">
    <p class="text-center p-5">Cargando detalles de la nómina...</p>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const csrfToken = '<?php echo $_SESSION['csrf_token']; ?>';
    const nominaId = <?php echo $nomina_id; ?>;
    const container = document.getElementById('nomina-detalle-container');
    const mainTitle = document.getElementById('main-title');
    const puedeGestionar = <?php echo puede('gestionar_nomina') ? 'true' : 'false'; ?>;

    const formatoMoneda = (valor, simbolo = '$') => `${simbolo}${ (parseFloat(valor) || 0).toFixed(2) }`;
    const formatoBs = (valor) => `Bs. ${ (parseFloat(valor) || 0).toLocaleString('es-VE', { minimumFractionDigits: 2, maximumFractionDigits: 2 }) }`;
    const formatDate = (dateString) => new Date(dateString + 'T00:00:00').toLocaleDateString('es-VE');
    
    // Formato para tasa BCV con 4 decimales exactos
    const formatoTasaBCV = (valor) => {
        return parseFloat(valor).toFixed(4);
    };

    // Array para trackear empleados pagados
    let empleadosPagados = new Set();

    const cargarDetalles = () => {
        fetch(`../api/nominas_ajax.php?action=obtener_detalle_nomina&id=${nominaId}`)
            .then(res => res.json())
            .then(response => {
                if (response.success) {
                    renderizarDetalles(response.data);
                } else {
                    container.innerHTML = `<div class="alert alert-danger">${response.message}</div>`;
                }
            })
            .catch(err => {
                container.innerHTML = `<div class="alert alert-danger">Error de conexión al cargar los detalles.</div>`;
            });
    };

    const toggleEmpleadoPagado = (empleadoId, checkbox) => {
        const fila = checkbox.closest('tr');
        
        if (checkbox.checked) {
            empleadosPagados.add(empleadoId);
            fila.classList.add('fila-pagada');
        } else {
            empleadosPagados.delete(empleadoId);
            fila.classList.remove('fila-pagada');
        }
    };

    const toggleTodosLosEmpleados = (checkbox) => {
        const checkboxesEmpleados = document.querySelectorAll('.checkbox-empleado');
        
        checkboxesEmpleados.forEach(cb => {
            const empleadoId = cb.getAttribute('data-empleado-id');
            const fila = cb.closest('tr');
            
            cb.checked = checkbox.checked;
            
            if (checkbox.checked) {
                empleadosPagados.add(empleadoId);
                fila.classList.add('fila-pagada');
            } else {
                empleadosPagados.delete(empleadoId);
                fila.classList.remove('fila-pagada');
            }
        });
    };

    const renderizarDetalles = (data) => {
        const { nomina, detalles } = data;
        
        mainTitle.textContent = `Detalle: ${nomina.nombre_nomina}`;

        let totalGeneralUSD = 0;
        let totalGeneralBS = 0;

        const filasTabla = detalles.map(d => {
            totalGeneralUSD += parseFloat(d.total_pagar);
            totalGeneralBS += parseFloat(d.total_pagar_bs);
            return `
                <tr>
                    <td>
                        ${puedeGestionar && nomina.estado === 'Generada' ? 
                            `<input type="checkbox" class="checkbox-empleado form-check-input" 
                                    data-empleado-id="${d.empleado_id}" 
                                    onchange="toggleEmpleadoPagado('${d.empleado_id}', this)">` 
                            : ''
                        }
                        ${d.nombre_empleado}
                    </td>
                    <td>${d.cedula}</td>
                    <td class="text-end">${formatoMoneda(d.total_asignaciones)}</td>
                    <td class="text-end">${formatoMoneda( (parseFloat(d.total_asignaciones)/2) )}</td>
                    <td class="text-end text-danger">${formatoMoneda(d.total_deducciones)}</td>
                    <td class="text-end text-primary fw-bold">${formatoMoneda(d.total_pagar)}</td>
                    <td class="text-end text-success fw-bold">${formatoBs(d.total_pagar_bs)}</td>
                </tr>
            `;
        }).join('');

        const botonPagarHtml = (puedeGestionar && nomina.estado === 'Generada')
            ? `<button class="btn btn-success btn-lg" id="btn-marcar-pagada"><i class="fa-solid fa-check-circle me-2"></i>Marcar como Pagada</button>`
            : '';
        
        const estadoBadge = (nomina.estado === 'Pagada')
            ? `<div class="value text-success"><i class="fa-solid fa-check-circle"></i> PAGADA</div>`
            : `<div class="value text-info">${nomina.estado.toUpperCase()}</div>`;

        const controlesSection = (puedeGestionar && nomina.estado === 'Generada') ? `
            <div class="mb-3">
                <input type="checkbox" id="toggle-todos" class="form-check-input me-2" onchange="toggleTodosLosEmpleados(this)">
                <label for="toggle-todos" class="form-check-label fw-bold">Seleccionar/Deseleccionar Todos</label>
            </div>
        ` : '';

        container.innerHTML = `
            <div class="detalle-header">
                <div class="detalle-info-box">
                    <div class="label">Nómina</div>
                    <div class="value">${nomina.nombre_nomina}</div>
                </div>
                <div class="detalle-info-box">
                    <div class="label">Periodo</div>
                    <div class="value">${formatDate(nomina.fecha_inicio)} al ${formatDate(nomina.fecha_fin)}</div>
                </div>
                <div class="detalle-info-box">
                    <div class="label">Tasa de Cambio (BCV)</div>
                    <div class="value">${formatoTasaBCV(nomina.tasa_cambio)}</div>
                </div>
                <div class="detalle-info-box">
                    <div class="label">Estado</div>
                    ${estadoBadge}
                </div>
            </div>

            <h5 class="detalle-titulo">Resumen de Pagos</h5>
            ${controlesSection}
            <div class="table-responsive">
                <table class="ami-table">
                    <thead>
                        <tr>
                            <th>Empleado</th>
                            <th>Cédula</th>
                            <th class="text-end">Total Asignaciones ($)</th>
                            <th class="text-end">Abono Quincenal ($)</th>
                            <th class="text-end">Total Deducciones ($)</th>
                            <th class="text-end">Total a Pagar ($)</th>
                            <th class="text-end">Total a Pagar (Bs.)</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${filasTabla}
                    </tbody>
                    <tfoot>
                        <tr>
                            <td colspan="5" class="text-end fw-bold">TOTALES GENERALES:</td>
                            <td class="text-end fw-bold fs-5 total-final-usd">${formatoMoneda(totalGeneralUSD)}</td>
                            <td class="text-end fw-bold fs-5 text-success">${formatoBs(totalGeneralBS)}</td>
                        </tr>
                    </tfoot>
                </table>
            </div>

            <div class="text-end mt-4">
                ${botonPagarHtml}
            </div>
        `;

        // Hacer las funciones globales para que funcionen con onclick
        window.toggleEmpleadoPagado = toggleEmpleadoPagado;
        window.toggleTodosLosEmpleados = toggleTodosLosEmpleados;

        // Añadir el listener solo si el botón existe
        if (botonPagarHtml) {
            document.getElementById('btn-marcar-pagada').addEventListener('click', marcarComoPagada);
        }
    };

    const marcarComoPagada = () => {
        Swal.fire({
            title: '¿Confirmar Acción?',
            text: "Esta acción marcará la nómina como pagada y no se podrá revertir.",
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#198754',
            cancelButtonColor: '#6c757d',
            confirmButtonText: 'Sí, confirmar pago',
            cancelButtonText: 'Cancelar'
        }).then((result) => {
            if (result.isConfirmed) {
                const btn = document.getElementById('btn-marcar-pagada');
                btn.disabled = true;
                btn.innerHTML = `<span class="spinner-border spinner-border-sm"></span> Procesando...`;

                const formData = new FormData();
                formData.append('action', 'marcar_como_pagada');
                formData.append('csrf_token', csrfToken);
                formData.append('nomina_id', nominaId);
                
                // Enviar también los empleados marcados individualmente
                formData.append('empleados_pagados', JSON.stringify(Array.from(empleadosPagados)));

                fetch('../api/nominas_ajax.php', { method: 'POST', body: formData })
                    .then(res => res.json())
                    .then(response => {
                        if (response.success) {
                            Swal.fire('¡Éxito!', response.message, 'success');
                            cargarDetalles(); // Recargar la vista para mostrar el nuevo estado
                        } else {
                            Swal.fire('Error', response.message, 'error');
                            btn.disabled = false;
                            btn.innerHTML = `<i class="fa-solid fa-check-circle me-2"></i>Marcar como Pagada`;
                        }
                    })
                    .catch(err => {
                         Swal.fire('Error de Conexión', 'No se pudo comunicar con el servidor.', 'error');
                         btn.disabled = false;
                         btn.innerHTML = `<i class="fa-solid fa-check-circle me-2"></i>Marcar como Pagada`;
                    });
            }
        });
    };

    // Carga inicial
    cargarDetalles();
});
</script>

<?php include_once 'templates/footer.php'; ?>