<?php
require_once 'includes/seguridad.php'; 

// Permiso para que el supervisor vea el detalle. Podríamos usar el mismo o uno más específico.
if (!puede('ver_reactivacion_supervisor')) {
    header('Location: inicio.php?error=acceso_denegado');
    exit();
}

$lote_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
if (!$lote_id) {
    header('Location: reactivacion-supervisor.php?error=lote_invalido');
    exit();
}

$page_title = 'Gestionar Lote de Reactivación'; // El título se actualizará dinámicamente
include_once 'templates/header.php';
?>

<style>
    .stat-card {
        background-color: rgba(0,0,0,0.2);
        padding: 1.5rem;
        border-radius: 8px;
        text-align: center;
    }
    .stat-card .stat-value {
        font-size: 2.5rem;
        font-weight: bold;
        color: var(--primary-color);
    }
    .stat-card .stat-label {
        font-size: 1rem;
        color: var(--light-text);
        opacity: 0.8;
    }
</style>

<div class="main-header">
    <div>
        <a href="reactivacion-supervisor.php" class="btn btn-outline-light mb-2"><i class="fa-solid fa-arrow-left"></i> Volver a Lotes</a>
        <h1 id="lote-nombre-titulo"><?php echo $page_title; ?></h1>
    </div>
</div>

<div class="row mb-4">
    <div class="col-md-3"><div class="stat-card"><div class="stat-value" id="stat-total">0</div><div class="stat-label">Total Clientes</div></div></div>
    <div class="col-md-3"><div class="stat-card"><div class="stat-value" id="stat-asignados">0</div><div class="stat-label">Asignados</div></div></div>
    <div class="col-md-3"><div class="stat-card"><div class="stat-value" id="stat-pendientes">0</div><div class="stat-label">Pendientes por Asignar</div></div></div>
    <div class="col-md-3"><div class="stat-card"><div class="stat-value" id="stat-contactados">0%</div><div class="stat-label">Progreso de Contacto</div></div></div>
</div>

<div class="info-card">
    <div class="toolbar d-flex justify-content-between align-items-center mb-3">
        <div>
            <label for="select-operador" class="me-2">Asignar seleccionados a:</label>
            <select id="select-operador" class="form-select form-select-sm d-inline-block w-auto"></select>
        </div>
        <?php if (puede('distribuir_lote_reactivacion')): ?>
        <button id="btn-asignar-seleccionados" class="btn btn-success" disabled>
            <i class="fa-solid fa-user-check me-2"></i>Asignar <span id="contador-seleccion">0</span> clientes
        </button>
        <?php endif; ?>
    </div>
    
    <table class="ami-table">
        <thead>
            <tr>
                <th><input type="checkbox" id="seleccionar-todos"></th>
                <th>Nombre del Cliente</th>
                <th>Cédula</th>
                <th>Teléfono</th>
                <th>Operador Asignado</th>
                <th>Estado del Contacto</th>
            </tr>
        </thead>
        <tbody id="tabla-clientes-lote">
            </tbody>
    </table>
</div>


<script>
document.addEventListener('DOMContentLoaded', function() {
    const loteId = <?php echo json_encode($lote_id); ?>;
    const csrfToken = '<?php echo $_SESSION['csrf_token']; ?>';
    const tbody = document.getElementById('tabla-clientes-lote');
    const selectOperador = document.getElementById('select-operador');
    const btnAsignar = document.getElementById('btn-asignar-seleccionados');
    const contadorSeleccion = document.getElementById('contador-seleccion');

    let clientesDelLote = [];

    function cargarDetallesLote() {
        tbody.innerHTML = '<tr><td colspan="6" class="text-center">Cargando detalles del lote...</td></tr>';

        Promise.all([
            fetch(`/api/reactivacion_ajax.php?action=get_detalles_lote_supervisor&lote_id=${loteId}`),
            fetch('/api/reactivacion_ajax.php?action=get_operadores_equipo') // Asume que el supervisor logueado es el que pide
        ])
        .then(responses => Promise.all(responses.map(res => res.json())))
        .then(([loteData, operadoresData]) => {
            if (!loteData.success) throw new Error(loteData.message);
            if (!operadoresData.success) throw new Error(operadoresData.message);

            document.getElementById('lote-nombre-titulo').textContent = `Gestionando Lote: ${loteData.data.info.nombre_lote}`;
            clientesDelLote = loteData.data.clientes;
            renderizarClientes(clientesDelLote);
            actualizarStats(loteData.data.stats);

            // Cargar operadores en el select
            selectOperador.innerHTML = '<option value="">-- Mi Equipo --</option>';
            operadoresData.data.forEach(op => {
                selectOperador.innerHTML += `<option value="${op.id}">${op.nombre}</option>`;
            });
        })
        .catch(error => {
            Swal.fire('Error', error.message, 'error');
            tbody.innerHTML = `<tr><td colspan="6" class="text-center text-danger">${error.message}</td></tr>`;
        });
    }

    function renderizarClientes(clientes) {
        if (clientes.length === 0) {
            tbody.innerHTML = '<tr><td colspan="6" class="text-center">Este lote no tiene clientes.</td></tr>';
            return;
        }

        let filasHtml = '';
        clientes.forEach(cliente => {
            let estadoClass = '';
            switch(cliente.estado_contacto) {
                case 'Pendiente': estadoClass = 'text-warning'; break;
                case 'Contactado': estadoClass = 'text-info'; break;
                case 'Agendado': estadoClass = 'text-success'; break;
                case 'No Interesado': estadoClass = 'text-danger'; break;
            }

            filasHtml += `
                <tr>
                    <td><input class="form-check-input" type="checkbox" value="${cliente.reactivacion_id}" data-cliente-id="${cliente.cliente_id}"></td>
                    <td><strong>${cliente.nombre}</strong></td>
                    <td>${cliente.cedula}</td>
                    <td>${cliente.telefono || 'N/A'}</td>
                    <td>${cliente.operador_nombre || '<span class="text-white-50">Sin asignar</span>'}</td>
                    <td><strong class="${estadoClass}">${cliente.estado_contacto}</strong></td>
                </tr>
            `;
        });
        tbody.innerHTML = filasHtml;
    }
    
    function actualizarStats(stats) {
        document.getElementById('stat-total').textContent = stats.total;
        document.getElementById('stat-asignados').textContent = stats.asignados;
        document.getElementById('stat-pendientes').textContent = stats.pendientes;
        document.getElementById('stat-contactados').textContent = `${stats.progreso}%`;
    }

    tbody.addEventListener('change', function(e) {
        if (e.target.matches('input[type="checkbox"]')) {
            if (e.target.id === 'seleccionar-todos') {
                const checkboxes = tbody.querySelectorAll('input[type="checkbox"][data-cliente-id]');
                checkboxes.forEach(cb => cb.checked = e.target.checked);
            }
            actualizarEstadoAsignacion();
        }
    });

    selectOperador.addEventListener('change', actualizarEstadoAsignacion);

    function actualizarEstadoAsignacion() {
        const checkboxes = tbody.querySelectorAll('input[type="checkbox"][data-cliente-id]:checked');
        const operadorSeleccionado = selectOperador.value;
        const numSeleccionados = checkboxes.length;

        contadorSeleccion.textContent = numSeleccionados;
        btnAsignar.disabled = numSeleccionados === 0 || !operadorSeleccionado;
    }

    btnAsignar.addEventListener('click', function() {
        const checkboxes = tbody.querySelectorAll('input[type="checkbox"][data-cliente-id]:checked');
        const reactivacionIds = Array.from(checkboxes).map(cb => cb.value);
        const operadorId = selectOperador.value;

        if (reactivacionIds.length === 0 || !operadorId) {
            Swal.fire('Atención', 'Debes seleccionar al menos un cliente y un operador.', 'warning');
            return;
        }

        const formData = new FormData();
        formData.append('action', 'asignar_clientes_operador');
        formData.append('csrf_token', csrfToken);
        formData.append('reactivacion_ids', JSON.stringify(reactivacionIds));
        formData.append('operador_id', operadorId);
        formData.append('lote_id', loteId);

        fetch('/api/reactivacion_ajax.php', {
            method: 'POST',
            body: formData
        })
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                Swal.fire('¡Éxito!', data.message, 'success');
                cargarDetallesLote(); // Recargar todo para ver los cambios
            } else {
                Swal.fire('Error', data.message, 'error');
            }
        });
    });

    cargarDetallesLote();
});
</script>

<?php include_once 'templates/footer.php'; ?>