<?php
require_once 'includes/seguridad.php';

// --- Validación de Acceso ---
$asesor_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
if (!$asesor_id) {
    header('Location: supervisor.php');
    exit();
}

// Verificar si el supervisor actual tiene permiso para ver a este asesor
$equipo_ids = [];
if (puede('supervisar_todos_operadores')) {
    // Si es superadmin, puede ver a cualquiera, solo validamos que el ID sea de un operador
    $stmt_check = $conn->prepare("SELECT u.id FROM usuarios u JOIN roles r ON u.rol_id = r.id WHERE u.id = ? AND r.clave = 'operador'");
    $stmt_check->execute([$asesor_id]);
    if (!$stmt_check->fetch()) {
        header('Location: supervisor.php?error=no_encontrado');
        exit();
    }
} else {
    // Si es supervisor normal, verificar que el asesor esté en su equipo
    $stmt_equipo = $conn->prepare("SELECT operador_id FROM equipo_supervisor WHERE supervisor_id = ?");
    $stmt_equipo->execute([$_SESSION['user_id']]);
    $equipo_ids = $stmt_equipo->fetchAll(PDO::FETCH_COLUMN);
    if (!in_array($asesor_id, $equipo_ids)) {
        header('Location: supervisor.php?error=acceso_denegado');
        exit();
    }
}

// Obtener nombre del asesor para el título
$stmt_asesor = $conn->prepare("SELECT nombre FROM usuarios WHERE id = ?");
$stmt_asesor->execute([$asesor_id]);
$nombre_asesor = $stmt_asesor->fetchColumn();

$page_title = 'Supervisando a: ' . htmlspecialchars($nombre_asesor);
include_once 'templates/header.php';
?>
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.css">
<style>
    .main-header { display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 1rem; margin-bottom: 1.5rem; }
    .toolbar { display: grid; grid-template-columns: repeat(auto-fit, minmax(220px, 1fr)); gap: 1rem; align-items: end; margin-bottom: 2rem; }
    .badge-estado { padding: 0.4em 0.7em; border-radius: 10px; font-weight: 500; text-transform: capitalize; color: white; }
    .telefono-revelado { font-weight: bold; font-style: normal; }
    .pagination-container { display: flex; justify-content: center; align-items: center; margin-top: 2rem; padding-top: 1.5rem; border-top: 1px solid var(--border-color); user-select: none; }
    .pagination-btn { background: rgba(255,255,255,0.05); border: 1px solid var(--border-color); color: var(--light-text); padding: 8px 14px; margin: 0 4px; border-radius: 8px; cursor: pointer; transition: all 0.2s ease; font-size: 0.9em; }
    .pagination-btn:hover { background-color: var(--primary-color); color: var(--dark-text); }
    .pagination-btn.active { background-color: var(--primary-color); color: var(--dark-text); font-weight: bold; }
    .pagination-btn.disabled { opacity: 0.4; cursor: not-allowed; }
    .pagination-info { margin: 0 1rem; font-size: 0.9em; opacity: 0.8; }
    @keyframes highlight-change { 0% { background-color: rgba(120, 227, 32, 0.3); } 100% { background-color: transparent; } }
    .fila-actualizada { animation: highlight-change 2s ease-out; }

    /* --- ESTILOS PARA NOTAS (NUEVO) --- */
    .ami-table th.col-notas, .ami-table td.col-notas {
        text-align: center;
        width: 60px;
    }
    .ami-table td.col-notas i {
        cursor: help;
        font-size: 1.1rem;
        color: #adb5bd;
        transition: color 0.2s ease;
    }
    .ami-table td.col-notas i:hover {
        color: var(--primary-color);
    }
</style>

<div class="main-header">
    <div>
        <a href="supervisor.php" class="btn btn-outline-light mb-2"><i class="fa-solid fa-arrow-left"></i> Volver al Dashboard</a>
        <h1><?php echo $page_title; ?></h1>
    </div>
    <div id="live-indicator" class="d-flex align-items-center gap-2 text-success">
        <i class="fa-solid fa-circle fa-beat"></i>
        <span><strong>EN VIVO</strong> (Actualizando cada 10s)</span>
    </div>
</div>

<div class="info-card">
    <div class="toolbar">
        <div><label class="ami-form-label">Buscar</label><input type="text" id="searchInput" class="form-control form-control-dark" placeholder="Escribe para buscar..."></div>
        <div><label class="ami-form-label">Campaña</label><select id="filtroCampana" class="form-select"><option value="">Todas</option><option>Total</option><option>Pelvica</option><option>Bioescaner</option></select></div>
        <div><label class="ami-form-label">Rango de Fechas</label><input type="text" id="filtroFecha" class="form-control" placeholder="Selecciona un rango..."></div>
    </div>
    <div class="table-responsive">
        <table class="ami-table">
            <thead>
                <tr>
                    <th>Nombre</th>
                    <th>Teléfono</th>
                    <th>Ciudad</th>
                    <th>Campaña</th>
                    <th>Estado</th>
                    <th>Última Acción</th>
                    <th class="col-notas" title="Historial de Notas"><i class="fa-solid fa-note-sticky"></i></th>
                    <th>Fecha Carga</th>
                </tr>
            </thead>
            <tbody id="prospectosTableBody"></tbody>
        </table>
    </div>
    <div id="pagination-controls" class="pagination-container"></div>
</div>

<script src="https://cdn.jsdelivr.net/npm/flatpickr"></script>
<script src="https://npmcdn.com/flatpickr/dist/l10n/es.js"></script>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const asesorId = <?php echo $asesor_id; ?>;
    let currentProspectos = new Map();
    let currentPage = 1;
    let currentPaginationData = {};
    let updateInterval;

    flatpickr("#filtroFecha", {
        mode: "range", dateFormat: "Y-m-d", altInput: true, altFormat: "d/m/Y", locale: "es",
        onClose: () => cargarProspectos(1)
    });

    function construirURL() {
        let url = `/api/supervisor_ajax.php?action=cargar_prospectos_asesor&asesor_id=${asesorId}&page=${currentPage}`;
        const campana = document.getElementById('filtroCampana').value;
        const fechaRangeStr = document.getElementById('filtroFecha')._flatpickr.input.value;
        if (campana) url += `&campana=${encodeURIComponent(campana)}`;
        if(fechaRangeStr.includes(' a ')) {
            const dates = fechaRangeStr.split(' a ');
            const fechaInicio = dates[0].split('/').reverse().join('-');
            const fechaFin = dates[1].split('/').reverse().join('-');
            url += `&fecha_inicio=${encodeURIComponent(fechaInicio)}&fecha_fin=${encodeURIComponent(fechaFin)}`;
        }
        return url;
    }

    function cargarProspectos(page = 1) {
        currentPage = page;
        const url = construirURL();

        fetch(url)
        .then(res => res.ok ? res.json() : Promise.reject(res))
        .then(data => {
            if (!data.success) {
                Swal.fire('Error', data.message, 'error');
                clearInterval(updateInterval);
                return;
            }
            currentPaginationData = data.pagination;
            renderizarTabla(data.data);
            renderizarPaginacion(data.pagination);
        })
        .catch(error => {
            console.error("Error de conexión:", error);
            document.getElementById('live-indicator').innerHTML = '<i class="fa-solid fa-circle text-danger"></i> <span><strong>CONEXIÓN PERDIDA</strong></span>';
            clearInterval(updateInterval);
        });
    }

    function renderizarTabla(nuevosProspectos) {
        const tablaBody = document.getElementById('prospectosTableBody');
        const filtro = document.getElementById('searchInput').value.toLowerCase();
        
        const prospectosFiltrados = nuevosProspectos.filter(p => 
            (p.nombre || '').toLowerCase().includes(filtro) ||
            (p.ciudad || '').toLowerCase().includes(filtro) ||
            (p.estado || '').toLowerCase().includes(filtro)
        );

        if (prospectosFiltrados.length === 0) {
            // COLSPAN ACTUALIZADO A 8 (MODIFICADO)
            tablaBody.innerHTML = '<tr><td colspan="8" class="text-center">No se encontraron prospectos.</td></tr>';
            currentProspectos.clear();
            return;
        }

        const newProspectosMap = new Map();
        prospectosFiltrados.forEach(p => {
            newProspectosMap.set(p.id.toString(), JSON.stringify(p));
        });

        tablaBody.innerHTML = prospectosFiltrados.map(p => {
            const fechaCarga = new Date(p.fecha_creacion).toLocaleDateString('es-ES');
            const telefonoHTML = p.telefono ? `<span class="telefono-revelado">${p.telefono}</span>` : 'N/A';
            const estadoColor = getColorPorEstado(p.estado);
            
            // LÓGICA PARA TOOLTIP DE NOTAS (NUEVO)
            const notasTooltip = (p.notas || 'Sin notas').replace(/"/g, '&quot;');
            const notasHTML = `<td class="col-notas"><i class="fa-solid fa-note-sticky" data-bs-toggle="tooltip" data-bs-placement="top" title="${notasTooltip}"></i></td>`;

            let highlightClass = '';
            const oldProspecto = currentProspectos.get(p.id.toString());
            if (oldProspecto && oldProspecto !== JSON.stringify(p)) {
                highlightClass = 'fila-actualizada';
            }

            return `
                <tr id="prospecto-${p.id}" class="${highlightClass}">
                    <td>${p.nombre}</td>
                    <td>${telefonoHTML}</td>
                    <td>${p.ciudad || 'N/A'}</td>
                    <td>${p.campana || 'N/A'}</td>
                    <td><span class="badge-estado" style="background-color: ${estadoColor.bg}; color: ${estadoColor.text};">${p.estado}</span></td>
                    <td>${p.ultima_accion || '---'}</td>
                    ${notasHTML} <td>${fechaCarga}</td>
                </tr>
            `;
        }).join('');

        currentProspectos = newProspectosMap;

        // INICIALIZAR TOOLTIPS DE BOOTSTRAP (NUEVO)
        const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
        tooltipTriggerList.map(function (tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl);
        });
    }

    function renderizarPaginacion(pagination) {
        // Idéntico a prospectos.php
        const container = document.getElementById('pagination-controls');
        if (!pagination || pagination.total_pages <= 1) {
            container.innerHTML = ''; return;
        }
        let html = `<span class="pagination-info">Página ${pagination.current_page} de ${pagination.total_pages}</span>`;
        html += `<button class="pagination-btn ${pagination.current_page === 1 ? 'disabled' : ''}" onclick="window.cargarPagina(${pagination.current_page - 1})">Anterior</button>`;
        let startPage = Math.max(1, pagination.current_page - 2);
        let endPage = Math.min(pagination.total_pages, pagination.current_page + 2);
        if (startPage > 1) html += `<button class="pagination-btn" onclick="window.cargarPagina(1)">1</button><span>...</span>`;
        for (let i = startPage; i <= endPage; i++) {
            html += `<button class="pagination-btn ${i === pagination.current_page ? 'active' : ''}" onclick="window.cargarPagina(${i})">${i}</button>`;
        }
        if (endPage < pagination.total_pages) html += `<span>...</span><button class="pagination-btn" onclick="window.cargarPagina(${pagination.total_pages})">${pagination.total_pages}</button>`;
        html += `<button class="pagination-btn ${pagination.current_page === pagination.total_pages ? 'disabled' : ''}" onclick="window.cargarPagina(${pagination.current_page + 1})">Siguiente</button>`;
        container.innerHTML = html;
    }

    window.cargarPagina = (page) => {
        if (page > 0 && page <= currentPaginationData.total_pages) {
            cargarProspectos(page);
        }
    };
    
    function getColorPorEstado(estado) {
        const colores = {'nuevo':{bg:'rgba(108,117,125,0.3)',text:'#ced4da'},'por confirmar':{bg:'rgba(13,202,240,0.3)',text:'#6edff6'},'agendado':{bg:'rgba(253,126,20,0.3)',text:'#fdb44b'},'asistido':{bg:'rgba(25,135,84,0.3)',text:'#75b798'},'no contesta':{bg:'rgba(255,193,7,0.4)',text:'#000'},'cancelo':{bg:'rgba(220,53,69,0.3)',text:'#f1aeb5'},'seguimiento':{bg:'rgba(102,16,242,0.3)',text:'#cda5fe'},'no interesado':{bg:'rgba(33,37,41,0.5)',text:'#adb5bd'}};
        return colores[estado] || colores['nuevo'];
    }

    document.getElementById('searchInput').addEventListener('input', () => cargarProspectos(1));
    document.getElementById('filtroCampana').addEventListener('change', () => cargarProspectos(1));

    cargarProspectos(1);
    updateInterval = setInterval(() => cargarProspectos(currentPage), 10000);
});
</script>

<?php include_once 'templates/footer.php'; ?>