<?php
define('IS_AJAX', true);
require_once '../administracion/includes/seguridad.php';

header('Content-Type: application/json');
$response = ['success' => false, 'message' => 'Acción no válida.'];
$usuario_id = $_SESSION['user_id'] ?? 0;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        $response['message'] = 'Error de seguridad (CSRF).';
        echo json_encode($response);
        exit();
    }
}

$action = $_POST['action'] ?? $_GET['action'] ?? '';

// --- Lógica de Rate Limiting (Protección Anti-Fuerza Bruta) ---
if (!function_exists('check_rate_limit')) {
    function check_rate_limit($key, $limit = 5, $lockout_time = 900) { // 15 minutos
        if (isset($_SESSION[$key]) && $_SESSION[$key]['attempts'] >= $limit) {
            if (time() - $_SESSION[$key]['time'] < $lockout_time) {
                return false; // Bloqueado
            }
            unset($_SESSION[$key]); // Tiempo de bloqueo expirado
        }
        return true;
    }
}
if (!function_exists('record_failed_attempt')) {
    function record_failed_attempt($key) {
        if (!isset($_SESSION[$key])) {
            $_SESSION[$key] = ['attempts' => 0, 'time' => 0];
        }
        $_SESSION[$key]['attempts']++;
        $_SESSION[$key]['time'] = time();
    }
}
if (!function_exists('clear_failed_attempts')) {
    function clear_failed_attempts($key) {
        if (isset($_SESSION[$key])) {
            unset($_SESSION[$key]);
        }
    }
}
// --- Fin de la Lógica de Rate Limiting ---


switch ($action) {
    case 'reautenticar_para_clave_maestra':
        if (!puede('gestionar_clave_maestra')) { $response['message'] = 'Acceso denegado.'; break; }

        $rate_limit_key = 'reauth_attempts_' . $usuario_id;
        if (!check_rate_limit($rate_limit_key)) {
            $response['message'] = 'Demasiados intentos fallidos. Por favor, inténtelo de nuevo en 15 minutos.';
            break;
        }

        $password = $_POST['password_actual_usuario'] ?? '';
        if (empty($password)) { $response['message'] = 'Debe ingresar su contraseña actual.'; break; }

        try {
            $stmt = $conn->prepare("SELECT password FROM usuarios WHERE id = ?");
            $stmt->execute([$usuario_id]);
            $hash_usuario = $stmt->fetchColumn();

            if ($hash_usuario && password_verify($password, $hash_usuario)) {
                // Autenticación exitosa, crear un token de tiempo limitado en la sesión
                $_SESSION['clave_maestra_unlocked'] = time();
                clear_failed_attempts($rate_limit_key);
                $response = ['success' => true, 'message' => 'Autenticación correcta. Ahora puede cambiar la clave maestra.'];
            } else {
                record_failed_attempt($rate_limit_key);
                $response['message'] = 'Su contraseña de usuario es incorrecta.';
            }
        } catch (PDOException $e) {
            $response['message'] = 'Error de base de datos: ' . $e->getMessage();
        }
        break;

    case 'cambiar_clave_maestra':
        if (!puede('gestionar_clave_maestra')) { $response['message'] = 'Acceso denegado.'; break; }

        // 1. Verificar si el usuario se ha re-autenticado recientemente
        if (!isset($_SESSION['clave_maestra_unlocked']) || (time() - $_SESSION['clave_maestra_unlocked'] > 300)) { // 5 minutos de validez
            unset($_SESSION['clave_maestra_unlocked']);
            $response['message'] = 'Su sesión de seguridad ha expirado. Por favor, vuelva a introducir su contraseña de usuario.';
            $response['requiere_reauth'] = true; // Flag para el frontend
            break;
        }

        // 2. Aplicar Rate Limiting a esta acción también
        $rate_limit_key = 'change_master_key_attempts_' . $usuario_id;
        if (!check_rate_limit($rate_limit_key)) {
            $response['message'] = 'Demasiados intentos fallidos con la clave maestra. Inténtelo de nuevo en 15 minutos.';
            break;
        }

        $clave_actual = $_POST['clave_actual'] ?? '';
        $nueva_clave = $_POST['nueva_clave'] ?? '';
        $confirmar_nueva_clave = $_POST['confirmar_nueva_clave'] ?? '';

        if (empty($clave_actual) || empty($nueva_clave) || empty($confirmar_nueva_clave)) { $response['message'] = 'Todos los campos son obligatorios.'; break; }
        if ($nueva_clave !== $confirmar_nueva_clave) { $response['message'] = 'La nueva clave y su confirmación no coinciden.'; break; }
        if (strlen($nueva_clave) < 8) { $response['message'] = 'La nueva clave debe tener al menos 8 caracteres.'; break; }

        try {
            // 3. Verificar la clave actual usando la nueva tabla 'claves_maestras'
            $stmt = $conn->prepare("SELECT id, clave_hash FROM claves_maestras WHERE usuario_id = ? AND estado = 'Activa'");
            $stmt->execute([$usuario_id]);
            $clave_maestra_info = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$clave_maestra_info || !password_verify($clave_actual, $clave_maestra_info['clave_hash'])) {
                record_failed_attempt($rate_limit_key);
                registrar_accion('cambio_clave_maestra_fallido', "El usuario ID {$usuario_id} intentó cambiar la clave maestra pero ingresó la clave actual incorrecta.");
                $response['message'] = 'La clave maestra actual que ingresaste es incorrecta.';
                break;
            }

            if (password_verify($nueva_clave, $clave_maestra_info['clave_hash'])) {
                $response['message'] = 'La nueva clave no puede ser igual a la actual.';
                break;
            }

            // 4. Si la clave es correcta, generar nuevo hash y actualizar
            $nuevo_hash = password_hash($nueva_clave, PASSWORD_DEFAULT);
            $stmt_update = $conn->prepare("UPDATE claves_maestras SET clave_hash = ?, ultimo_uso = NOW() WHERE id = ?");
            $stmt_update->execute([$nuevo_hash, $clave_maestra_info['id']]);

            // 5. Invalidar sesión de seguridad y limpiar contadores de intentos
            unset($_SESSION['clave_maestra_unlocked']);
            clear_failed_attempts($rate_limit_key);

            registrar_accion('cambio_clave_maestra_exitoso', "El usuario ID {$usuario_id} cambió su clave maestra de administrador exitosamente.");
            $response = ['success' => true, 'message' => '¡Clave maestra actualizada exitosamente!'];

        } catch (PDOException $e) {
            $response['message'] = 'Error de base de datos: ' . $e->getMessage();
        }
        break;

    default:
        $response['message'] = 'Acción de ajustes desconocida.';
        break;
}

echo json_encode($response);
?>

