<?php
define('IS_AJAX', true);
require_once '../administracion/includes/seguridad.php';

header('Content-Type: application/json');
$response = ['success' => false, 'message' => 'Acción no válida.'];
$usuario_id = $_SESSION['user_id'] ?? 0;

if ($_SERVER['REQUEST_METHOD'] === 'POST' && (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token']))) {
    $response['message'] = 'Error de seguridad (CSRF).';
    echo json_encode($response);
    exit();
}

$action = $_POST['action'] ?? $_GET['action'] ?? '';

switch ($action) {
    case 'cargar_dashboard':
        if (!puede('ver_inventario')) { $response['message'] = 'Acceso denegado.'; break; }
        try {
            function getStockParaRutas($conn, $rutas) {
                $stock_data = [];
                $stmt_stock_ruta = $conn->prepare("
                    SELECT p.sku, p.nombre, rs.cantidad 
                    FROM ruta_stock rs
                    JOIN almacen_productos p ON rs.producto_id = p.id
                    WHERE rs.ruta_id = ? ORDER BY p.nombre ASC
                ");
                foreach ($rutas as $ruta) {
                    $stmt_stock_ruta->execute([$ruta['id']]);
                    $productos_en_ruta = $stmt_stock_ruta->fetchAll(PDO::FETCH_ASSOC);
                    if (!empty($productos_en_ruta) || !$ruta['archivada']) {
                        $stock_data[] = [
                            'id' => $ruta['id'],
                            'nombre_ruta' => $ruta['nombre_ruta'],
                            'productos' => $productos_en_ruta
                        ];
                    }
                }
                return $stock_data;
            }

            $stmt_stats = $conn->query("SELECT (SELECT COUNT(*) FROM almacen_productos) as total_productos, (SELECT SUM(cantidad) FROM almacen_stock) as unidades_totales, (SELECT COUNT(*) FROM almacen_stock s JOIN almacen_productos p ON s.producto_id = p.id WHERE s.cantidad <= p.stock_minimo AND p.stock_minimo > 0) as productos_bajo_stock");
            $stats = $stmt_stats->fetch(PDO::FETCH_ASSOC);

            $stmt_inventario = $conn->query("SELECT p.id, p.sku, p.nombre, p.descripcion, p.categoria, p.stock_minimo, s.cantidad FROM almacen_productos p LEFT JOIN almacen_stock s ON p.id = s.producto_id ORDER BY p.nombre ASC");
            $inventario = $stmt_inventario->fetchAll(PDO::FETCH_ASSOC);

            $stmt_movimientos = $conn->query("SELECT m.fecha_movimiento, p.nombre as producto_nombre, m.tipo_movimiento, m.cantidad, m.motivo, u.nombre as usuario_nombre FROM almacen_movimientos m JOIN almacen_productos p ON m.producto_id = p.id LEFT JOIN usuarios u ON m.usuario_id = u.id ORDER BY m.fecha_movimiento DESC LIMIT 50");
            $movimientos = $stmt_movimientos->fetchAll(PDO::FETCH_ASSOC);

            $stmt_rutas_activas = $conn->query("SELECT id, nombre_ruta, 0 as archivada FROM rutas WHERE archivada = 0 ORDER BY nombre_ruta ASC");
            $rutas_activas = $stmt_rutas_activas->fetchAll(PDO::FETCH_ASSOC);
            $stock_rutas_activas = getStockParaRutas($conn, $rutas_activas);
            
            $stmt_rutas_archivadas = $conn->query("
                SELECT DISTINCT r.id, r.nombre_ruta, 1 as archivada 
                FROM rutas r
                JOIN ruta_stock rs ON r.id = rs.ruta_id
                WHERE r.archivada = 1 AND rs.cantidad > 0
                ORDER BY r.nombre_ruta ASC
            ");
            $rutas_archivadas = $stmt_rutas_archivadas->fetchAll(PDO::FETCH_ASSOC);
            $stock_rutas_archivadas = getStockParaRutas($conn, $rutas_archivadas);

            $response = ['success' => true, 'data' => [
                'stats' => $stats, 
                'inventario' => $inventario, 
                'movimientos' => $movimientos,
                'stock_rutas_activas' => $stock_rutas_activas,
                'stock_rutas_archivadas' => $stock_rutas_archivadas
            ]];
        } catch (Exception $e) { $response['message'] = 'Error al cargar los datos del almacén.'; error_log($e->getMessage()); }
        break;

    case 'obtener_producto':
        if (!puede('ver_inventario')) { $response['message'] = 'Acceso denegado.'; break; }
        $id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
        if(!$id) { $response['message'] = 'ID de producto no válido.'; break; }
        $stmt = $conn->prepare("SELECT * FROM almacen_productos WHERE id = ?");
        $stmt->execute([$id]);
        $producto = $stmt->fetch(PDO::FETCH_ASSOC);
        $response = $producto ? ['success' => true, 'data' => $producto] : ['message' => 'Producto no encontrado.'];
        break;

    case 'guardar_producto':
        if (!puede('gestionar_almacen')) { $response['message'] = 'Acceso denegado.'; break; }
        $id = filter_input(INPUT_POST, 'producto_id', FILTER_VALIDATE_INT);
        $nombre = trim($_POST['nombre'] ?? '');
        if (empty($nombre)) { $response['message'] = 'El nombre del producto es obligatorio.'; break; }

        try {
            $conn->beginTransaction();
            if ($id) {
                $stmt = $conn->prepare("UPDATE almacen_productos SET nombre=?, descripcion=?, sku=?, categoria=?, stock_minimo=? WHERE id=?");
                $stmt->execute([$nombre, $_POST['descripcion'], $_POST['sku'], $_POST['categoria'], $_POST['stock_minimo'], $id]);
                $message = 'Producto actualizado correctamente.';
            } else {
                $stmt = $conn->prepare("INSERT INTO almacen_productos (nombre, descripcion, sku, categoria, stock_minimo, creado_por) VALUES (?, ?, ?, ?, ?, ?)");
                $stmt->execute([$nombre, $_POST['descripcion'], $_POST['sku'], $_POST['categoria'], $_POST['stock_minimo'], $usuario_id]);
                $producto_id = $conn->lastInsertId();
                
                $stmt_stock = $conn->prepare("INSERT INTO almacen_stock (producto_id, cantidad) VALUES (?, 0)");
                $stmt_stock->execute([$producto_id]);
                $message = 'Producto creado correctamente.';
            }
            $conn->commit();
            $response = ['success' => true, 'message' => $message];
        } catch (PDOException $e) {
            $conn->rollBack();
            if ($e->getCode() == '23000') { $response['message'] = 'Error: El SKU ya existe para otro producto.'; }
            else { $response['message'] = 'Error de base de datos.'; error_log($e->getMessage()); }
        }
        break;

    case 'registrar_movimiento':
        if (!puede('gestionar_almacen')) { $response['message'] = 'Acceso denegado.'; break; }
        $producto_id = filter_input(INPUT_POST, 'producto_id_mov', FILTER_VALIDATE_INT);
        $tipo = $_POST['tipo_movimiento'] ?? '';
        $cantidad = filter_input(INPUT_POST, 'cantidad', FILTER_VALIDATE_INT);
        $motivo = trim($_POST['motivo'] ?? '');

        if (!$producto_id || !$cantidad || empty($motivo) || !in_array($tipo, ['ingreso', 'egreso', 'ajuste'])) {
            $response['message'] = 'Todos los campos son obligatorios.';
            break;
        }

        try {
            $conn->beginTransaction();
            if ($tipo === 'egreso') {
                $stmt_stock = $conn->prepare("SELECT cantidad FROM almacen_stock WHERE producto_id = ? FOR UPDATE");
                $stmt_stock->execute([$producto_id]);
                $stock_actual = $stmt_stock->fetchColumn();
                if ($stock_actual < $cantidad) {
                    $conn->rollBack();
                    $response['message'] = 'Error: Stock insuficiente en Almacén Central. Stock actual: ' . $stock_actual;
                    echo json_encode($response); exit();
                }
            }
            
            $stmt_mov = $conn->prepare("INSERT INTO almacen_movimientos (producto_id, tipo_movimiento, cantidad, motivo, usuario_id) VALUES (?, ?, ?, ?, ?)");
            $stmt_mov->execute([$producto_id, $tipo, $cantidad, $motivo, $usuario_id]);

            if ($tipo === 'ingreso') $sql_update = "UPDATE almacen_stock SET cantidad = cantidad + ? WHERE producto_id = ?";
            if ($tipo === 'egreso') $sql_update = "UPDATE almacen_stock SET cantidad = cantidad - ? WHERE producto_id = ?";
            if ($tipo === 'ajuste') $sql_update = "UPDATE almacen_stock SET cantidad = ? WHERE producto_id = ?";
            
            $stmt_update_stock = $conn->prepare($sql_update);
            $stmt_update_stock->execute([$cantidad, $producto_id]);

            $conn->commit();
            $response['success'] = true;
        } catch (Exception $e) {
            $conn->rollBack();
            $response['message'] = 'Error al registrar el movimiento.';
            error_log($e->getMessage());
        }
        break;

    case 'get_productos_para_transferencia':
        if (!puede('gestionar_almacen')) { $response['message'] = 'Acceso denegado.'; break; }
        try {
            $stmt = $conn->query("SELECT p.id, p.nombre, s.cantidad FROM almacen_productos p JOIN almacen_stock s ON p.id = s.producto_id WHERE s.cantidad > 0 ORDER BY p.nombre ASC");
            $productos = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $response = ['success' => true, 'data' => $productos];
        } catch (Exception $e) {
            $response['message'] = 'Error al cargar productos.';
        }
        break;
        
    case 'transferir_stock_a_ruta':
        if (!puede('gestionar_almacen')) {
            $response['message'] = 'Acceso denegado.';
            break;
        }

        $ruta_id = filter_input(INPUT_POST, 'ruta_id_trans', FILTER_VALIDATE_INT);
        $productos_json = $_POST['productos'] ?? '[]';
        $productos = json_decode($productos_json, true);

        if (!$ruta_id || empty($productos) || !is_array($productos)) {
            $response['message'] = 'Datos de transferencia incompletos o incorrectos.';
            break;
        }

        try {
            $conn->beginTransaction();

            $stmt_despacho = $conn->prepare("INSERT INTO almacen_despachos (ruta_id, usuario_id) VALUES (?, ?)");
            $stmt_despacho->execute([$ruta_id, $usuario_id]);
            $despacho_id = $conn->lastInsertId();

            $stmt_ruta_nombre = $conn->prepare("SELECT nombre_ruta FROM rutas WHERE id = ?");
            $stmt_ruta_nombre->execute([$ruta_id]);
            $nombre_ruta = $stmt_ruta_nombre->fetchColumn();
            
            foreach ($productos as $producto) {
                $producto_id = filter_var($producto['producto_id'], FILTER_VALIDATE_INT);
                $cantidad = filter_var($producto['cantidad'], FILTER_VALIDATE_INT);

                if (!$producto_id || !$cantidad || $cantidad <= 0) {
                    throw new Exception("Datos de producto inválidos en la lista.");
                }

                $stmt_check = $conn->prepare("SELECT cantidad FROM almacen_stock WHERE producto_id = ? FOR UPDATE");
                $stmt_check->execute([$producto_id]);
                $stock_central = $stmt_check->fetchColumn();

                if ($stock_central === false || $stock_central < $cantidad) {
                    throw new Exception("Stock insuficiente para el producto ID {$producto_id}. Disponible: " . ($stock_central ?: 0));
                }

                $stmt_debit = $conn->prepare("UPDATE almacen_stock SET cantidad = cantidad - ? WHERE producto_id = ?");
                $stmt_debit->execute([$cantidad, $producto_id]);
                
                $stmt_check_ruta = $conn->prepare("SELECT id FROM ruta_stock WHERE ruta_id = ? AND producto_id = ?");
                $stmt_check_ruta->execute([$ruta_id, $producto_id]);
                if ($ruta_stock_id = $stmt_check_ruta->fetchColumn()) {
                    $stmt_credit = $conn->prepare("UPDATE ruta_stock SET cantidad = cantidad + ? WHERE id = ?");
                    $stmt_credit->execute([$cantidad, $ruta_stock_id]);
                } else {
                    $stmt_credit = $conn->prepare("INSERT INTO ruta_stock (ruta_id, producto_id, cantidad) VALUES (?, ?, ?)");
                    $stmt_credit->execute([$ruta_id, $producto_id, $cantidad]);
                }

                $motivo = "Despacho #{$despacho_id} a ruta: " . $nombre_ruta;
                $stmt_mov = $conn->prepare("INSERT INTO almacen_movimientos (producto_id, tipo_movimiento, cantidad, motivo, usuario_id) VALUES (?, 'egreso', ?, ?, ?)");
                $stmt_mov->execute([$producto_id, $cantidad, $motivo, $usuario_id]);
                
                $stmt_item = $conn->prepare("INSERT INTO almacen_despacho_items (despacho_id, producto_id, cantidad_enviada) VALUES (?, ?, ?)");
                $stmt_item->execute([$despacho_id, $producto_id, $cantidad]);
            }

            $conn->commit();
            $response = ['success' => true, 'message' => 'Despacho generado correctamente.', 'despacho_id' => $despacho_id];

        } catch (Exception $e) {
            $conn->rollBack();
            $response['message'] = 'Error al procesar el despacho: ' . $e->getMessage();
            error_log($e->getMessage());
        }
        break;
    
    case 'get_rutas_activas':
        if (!puede('gestionar_almacen')) { $response['message'] = 'Acceso denegado.'; break; }
        try {
            $stmt = $conn->query("SELECT id, nombre_ruta FROM rutas WHERE activa = 1 AND archivada = 0 ORDER BY nombre_ruta ASC");
            $rutas = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $response = ['success' => true, 'data' => $rutas];
        } catch(PDOException $e) { $response['message'] = 'Error al cargar las rutas activas.'; }
        break;

    case 'reubicar_stock_ruta':
        if (!puede('gestionar_almacen')) { $response['message'] = 'Acceso denegado.'; break; }
        
        $ruta_origen_id = filter_input(INPUT_POST, 'ruta_origen_id', FILTER_VALIDATE_INT);
        $ruta_destino_id = filter_input(INPUT_POST, 'ruta_destino_id', FILTER_VALIDATE_INT);

        if (!$ruta_origen_id || !$ruta_destino_id) {
            $response['message'] = 'Debe seleccionar una ruta de origen y una de destino.';
            break;
        }
        if ($ruta_origen_id === $ruta_destino_id) {
            $response['message'] = 'La ruta de origen y destino no pueden ser la misma.';
            break;
        }

        try {
            $conn->beginTransaction();

            $stmt_nombres = $conn->prepare("SELECT id, nombre_ruta FROM rutas WHERE id = ? OR id = ?");
            $stmt_nombres->execute([$ruta_origen_id, $ruta_destino_id]);
            $nombres = $stmt_nombres->fetchAll(PDO::FETCH_KEY_PAIR);
            $nombre_origen = $nombres[$ruta_origen_id] ?? 'ID '.$ruta_origen_id;
            $nombre_destino = $nombres[$ruta_destino_id] ?? 'ID '.$ruta_destino_id;

            $stmt_stock_origen = $conn->prepare("SELECT producto_id, cantidad FROM ruta_stock WHERE ruta_id = ? AND cantidad > 0");
            $stmt_stock_origen->execute([$ruta_origen_id]);
            $stock_a_mover = $stmt_stock_origen->fetchAll(PDO::FETCH_ASSOC);

            if (empty($stock_a_mover)) {
                $conn->rollBack();
                $response['message'] = 'La ruta de origen no tiene stock para reubicar.';
                break;
            }

            $stmt_update_destino = $conn->prepare("UPDATE ruta_stock SET cantidad = cantidad + ? WHERE ruta_id = ? AND producto_id = ?");
            $stmt_insert_destino = $conn->prepare("INSERT INTO ruta_stock (ruta_id, producto_id, cantidad) VALUES (?, ?, ?)");
            $stmt_check_destino = $conn->prepare("SELECT id FROM ruta_stock WHERE ruta_id = ? AND producto_id = ?");
            $stmt_mov = $conn->prepare("INSERT INTO almacen_movimientos (producto_id, tipo_movimiento, cantidad, motivo, usuario_id) VALUES (?, ?, ?, ?, ?)");
            
            foreach ($stock_a_mover as $item) {
                $producto_id = $item['producto_id'];
                $cantidad = $item['cantidad'];
                
                $stmt_check_destino->execute([$ruta_destino_id, $producto_id]);
                if ($stmt_check_destino->fetch()) {
                    $stmt_update_destino->execute([$cantidad, $ruta_destino_id, $producto_id]);
                } else {
                    $stmt_insert_destino->execute([$ruta_destino_id, $producto_id, $cantidad]);
                }

                $motivo_egreso = "Transferencia desde ruta '{$nombre_origen}'";
                $stmt_mov->execute([$producto_id, 'egreso', $cantidad, $motivo_egreso, $usuario_id]);
                
                $motivo_ingreso = "Transferencia hacia ruta '{$nombre_destino}'";
                $stmt_mov->execute([$producto_id, 'ingreso', $cantidad, $motivo_ingreso, $usuario_id]);
            }
            
            $stmt_delete_origen = $conn->prepare("DELETE FROM ruta_stock WHERE ruta_id = ?");
            $stmt_delete_origen->execute([$ruta_origen_id]);
            
            $conn->commit();
            $response = ['success' => true, 'message' => 'Stock reubicado correctamente.'];

        } catch (Exception $e) {
            $conn->rollBack();
            $response['message'] = 'Error al reubicar el stock: ' . $e->getMessage();
            error_log($e->getMessage());
        }
        break;

    default:
        $response['message'] = 'Acción de almacén desconocida.';
        break;
}

echo json_encode($response);
?>