<?php
define('IS_AJAX', true);
require_once '../administracion/includes/seguridad.php';

header('Content-Type: application/json');
$response = ['success' => false, 'message' => 'Acción no válida.'];
$usuario_id = $_SESSION['user_id'] ?? 0;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        $response['message'] = 'Error de seguridad (CSRF).';
        echo json_encode($response);
        exit();
    }
}

$action = $_POST['action'] ?? $_GET['action'] ?? '';

switch ($action) {
    case 'get_jornada_coordinador':
        $jornada_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
        if (!$jornada_id) { 
            $response['message'] = 'ID de jornada no válido.'; 
            break; 
        }
        
        try {
            // ✅ CORRECCIÓN CLAVE: Verificar si es superadmin usando JOIN con roles.clave
            $stmt_check_admin = $conn->prepare("
                SELECT r.clave 
                FROM usuarios u
                JOIN roles r ON u.rol_id = r.id
                WHERE u.id = ?
            ");
            $stmt_check_admin->execute([$usuario_id]);
            $rol_clave = $stmt_check_admin->fetchColumn();
            $es_superadmin = ($rol_clave !== null && strtolower($rol_clave) === 'superadmin');

            // Lógica de seguridad para verificar acceso del coordinador
            $stmt_check = $conn->prepare("SELECT j.id, j.fecha_jornada, r.coordinador_id FROM jornadas j JOIN rutas r ON j.ruta_id = r.id WHERE j.id = ?");
            $stmt_check->execute([$jornada_id]);
            $jornada_info = $stmt_check->fetch(PDO::FETCH_ASSOC);
            
            if (!$jornada_info) {
                $response['message'] = 'Jornada no encontrada.';
                echo json_encode($response); exit();
            }
            
            // Si NO es superadmin, aplicar las restricciones normales
            if (!$es_superadmin) {
                if ($jornada_info['coordinador_id'] != $usuario_id || $jornada_info['fecha_jornada'] != date('Y-m-d')) {
                    $response['message'] = 'Acceso no autorizado a esta jornada.';
                    echo json_encode($response); exit();
                }
            }
            // Si ES superadmin, continúa sin restricciones

            // Consulta principal de pacientes, incluyendo TODOS los nuevos campos de pago, edad y sexo
            $stmt_pacientes = $conn->prepare("
                SELECT 
                    c.id as cliente_id, c.nombre, c.cedula, c.telefono, 
                    jc.id as jornada_cliente_id, jc.estado_cita, jc.total_facturado, 
                    jc.metodo_pago, jc.comprobante_pago, jc.notas_coordinador, jc.sexo, jc.edad,
                    jc.pago_divisas, jc.pago_bs_efectivo, jc.pago_pesos, jc.pago_zelle, jc.pago_punto, jc.pago_movil
                FROM jornada_clientes jc 
                JOIN clientes c ON jc.cliente_id = c.id 
                WHERE jc.jornada_id = ?
            ");
            $stmt_pacientes->execute([$jornada_id]);
            $pacientes = $stmt_pacientes->fetchAll(PDO::FETCH_ASSOC);

            $stmt_consumo = $conn->prepare("SELECT producto_id, cantidad_usada FROM jornada_consumo WHERE jornada_cliente_id = ?");
            $stmt_examenes = $conn->prepare("SELECT nombre_examen, precio FROM jornada_examenes WHERE jornada_cliente_id = ?");
            $stmt_medicamentos = $conn->prepare("SELECT producto_id, cantidad, precio FROM jornada_medicamentos WHERE jornada_cliente_id = ?");

            foreach($pacientes as &$paciente) {
                $paciente_jornada_id = $paciente['jornada_cliente_id'];
                $stmt_consumo->execute([$paciente_jornada_id]);
                $paciente['consumo'] = $stmt_consumo->fetchAll(PDO::FETCH_ASSOC);
                $stmt_examenes->execute([$paciente_jornada_id]);
                $paciente['examenes'] = $stmt_examenes->fetchAll(PDO::FETCH_ASSOC);
                $stmt_medicamentos->execute([$paciente_jornada_id]);
                $paciente['medicamentos'] = $stmt_medicamentos->fetchAll(PDO::FETCH_ASSOC);
            }

            // Agregar información adicional para superadmin
            $response_data = ['pacientes' => $pacientes];
            if ($es_superadmin) {
                $response_data['superadmin_access'] = true;
                $response_data['jornada_info'] = [
                    'fecha_jornada' => $jornada_info['fecha_jornada'],
                    'coordinador_id' => $jornada_info['coordinador_id'],
                    'es_fecha_actual' => $jornada_info['fecha_jornada'] === date('Y-m-d')
                ];
            }

            $response = ['success' => true, 'data' => $response_data];
            
        } catch(PDOException $e) { 
            $response['message'] = 'Error al cargar pacientes: ' . $e->getMessage(); 
            error_log("Error en get_jornada_coordinador: " . $e->getMessage()); 
        }
        break;

    case 'get_stock_ruta':
        $ruta_id = filter_input(INPUT_GET, 'ruta_id', FILTER_VALIDATE_INT);
        if (!$ruta_id) { $response['message'] = 'ID de ruta no válido.'; break; }
        try {
            $stmt = $conn->prepare("
                SELECT rs.producto_id, rs.cantidad, p.nombre, p.sku, p.categoria
                FROM ruta_stock rs
                JOIN almacen_productos p ON rs.producto_id = p.id
                WHERE rs.ruta_id = ?
            ");
            $stmt->execute([$ruta_id]);
            $stock = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $response = ['success' => true, 'data' => $stock];
        } catch (Exception $e) {
            $response['message'] = 'Error al obtener el stock de la ruta: ' . $e->getMessage();
        }
        break;

    case 'actualizar_paciente_jornada':
    $jornada_cliente_id = filter_input(INPUT_POST, 'jornada_cliente_id', FILTER_VALIDATE_INT);
    $jornada_id = filter_input(INPUT_POST, 'jornada_id', FILTER_VALIDATE_INT);
    if (!$jornada_cliente_id || !$jornada_id) { 
        $response['message'] = 'ID de paciente o jornada no válido.'; 
        break; 
    }

    try {
        // ✅ Verificar si es superadmin
        $stmt_check_admin = $conn->prepare("
            SELECT r.clave 
            FROM usuarios u
            JOIN roles r ON u.rol_id = r.id
            WHERE u.id = ?
        ");
        $stmt_check_admin->execute([$usuario_id]);
        $rol_clave = $stmt_check_admin->fetchColumn();
        $es_superadmin = ($rol_clave !== null && strtolower($rol_clave) === 'superadmin');

        // Si no es superadmin, verificar acceso normal
        if (!$es_superadmin) {
            $stmt_verify = $conn->prepare("
                SELECT j.fecha_jornada, r.coordinador_id 
                FROM jornadas j 
                JOIN rutas r ON j.ruta_id = r.id 
                WHERE j.id = ?
            ");
            $stmt_verify->execute([$jornada_id]);
            $jornada_verify = $stmt_verify->fetch(PDO::FETCH_ASSOC);
            
            if (!$jornada_verify || 
                $jornada_verify['coordinador_id'] != $usuario_id || 
                $jornada_verify['fecha_jornada'] != date('Y-m-d')) {
                $response['message'] = 'Acceso no autorizado para actualizar esta jornada.';
                break;
            }
        }

        $conn->beginTransaction();

        $stmt_ruta = $conn->prepare("SELECT ruta_id FROM jornadas WHERE id = ?");
        $stmt_ruta->execute([$jornada_id]);
        $ruta_id = $stmt_ruta->fetchColumn();
        
        $diferencias_stock = [];

        // --- LÓGICA PARA SUBIR COMPROBANTE DE PAGO ---
        $comprobante_filename = null;
        if (isset($_FILES['comprobante_pago']) && $_FILES['comprobante_pago']['error'] == 0) {
            $upload_dir = __DIR__ . '/../uploads/comprobantes/';
            if (!is_dir($upload_dir)) { mkdir($upload_dir, 0755, true); }
            
            $stmt_nombres = $conn->prepare("SELECT r.nombre_ruta, j.fecha_jornada, c.nombre FROM jornada_clientes jc JOIN jornadas j ON jc.jornada_id = j.id JOIN rutas r ON j.ruta_id = r.id JOIN clientes c ON jc.cliente_id = c.id WHERE jc.id = ?");
            $stmt_nombres->execute([$jornada_cliente_id]);
            $nombres = $stmt_nombres->fetch(PDO::FETCH_ASSOC);

            $nombre_ruta_saneado = preg_replace('/[^a-zA-Z0-9-]/', '-', $nombres['nombre_ruta']);
            $nombre_paciente_saneado = preg_replace('/[^a-zA-Z0-9-]/', '-', $nombres['nombre']);
            $fecha = (new DateTime($nombres['fecha_jornada']))->format('Y-m-d');
            $codigo_unico = uniqid();
            
            $extension = strtolower(pathinfo($_FILES['comprobante_pago']['name'], PATHINFO_EXTENSION));
            $comprobante_filename = "{$nombre_ruta_saneado}_{$fecha}_{$nombre_paciente_saneado}_{$codigo_unico}.{$extension}";
            
            if (!move_uploaded_file($_FILES['comprobante_pago']['tmp_name'], $upload_dir . $comprobante_filename)) {
                throw new Exception('Error al mover el archivo del comprobante.');
            }
        }

        // --- 1. PROCESAR EXÁMENES Y MEDICAMENTOS PARA CALCULAR EL TOTAL ---
        $total_facturado = 0;
        
        $examenes_nuevos = $_POST['examenes'] ?? [];
        foreach ($examenes_nuevos as $examen) {
            if (!empty($examen['tipo']) && isset($examen['precio'])) {
                $total_facturado += floatval($examen['precio']);
            }
        }
        
        $medicamentos_nuevos = $_POST['medicamentos'] ?? [];
        foreach($medicamentos_nuevos as $med) {
            if (!empty($med['producto_id']) && !empty($med['cantidad'])) {
                $total_facturado += floatval($med['cantidad']) * floatval($med['precio'] ?? 0);
            }
        }

        // --- 2. PROCESAR TABLAS DE EXÁMENES Y MEDICAMENTOS ---
        $conn->prepare("DELETE FROM jornada_examenes WHERE jornada_cliente_id = ?")->execute([$jornada_cliente_id]);
        $stmt_examen = $conn->prepare("INSERT INTO jornada_examenes (jornada_cliente_id, nombre_examen, precio) VALUES (?, ?, ?)");
        foreach ($examenes_nuevos as $examen) {
            if (!empty($examen['tipo']) && isset($examen['precio'])) {
                $stmt_examen->execute([$jornada_cliente_id, $examen['tipo'], floatval($examen['precio'])]);
            }
        }
        
        $stmt_meds_antiguos = $conn->prepare("SELECT producto_id, cantidad FROM jornada_medicamentos WHERE jornada_cliente_id = ?");
        $stmt_meds_antiguos->execute([$jornada_cliente_id]);
        $meds_antiguos = $stmt_meds_antiguos->fetchAll(PDO::FETCH_KEY_PAIR);
        $conn->prepare("DELETE FROM jornada_medicamentos WHERE jornada_cliente_id = ?")->execute([$jornada_cliente_id]);
        
        $stmt_insert_med = $conn->prepare("INSERT INTO jornada_medicamentos (jornada_cliente_id, producto_id, nombre_medicamento, cantidad, precio) VALUES (?, ?, ?, ?, ?)");
        $stmt_producto_nombre = $conn->prepare("SELECT nombre FROM almacen_productos WHERE id = ?");

        foreach($medicamentos_nuevos as $med) {
            if (empty($med['producto_id']) || empty($med['cantidad'])) continue;
            $producto_id = (int)$med['producto_id'];
            $cantidad_nueva = (int)$med['cantidad'];
            $precio_unitario = floatval($med['precio'] ?? 0);
            
            $stmt_producto_nombre->execute([$producto_id]);
            $nombre_medicamento = $stmt_producto_nombre->fetchColumn();

            $stmt_insert_med->execute([$jornada_cliente_id, $producto_id, $nombre_medicamento, $cantidad_nueva, $precio_unitario]);
            
            $cantidad_antigua = (int)($meds_antiguos[$producto_id] ?? 0);
            $diferencia = $cantidad_antigua - $cantidad_nueva;
            $diferencias_stock[$producto_id] = ($diferencias_stock[$producto_id] ?? 0) + $diferencia;
        }
        foreach ($meds_antiguos as $producto_id => $cantidad_antigua) {
            $encontrado = false;
            if(!empty($medicamentos_nuevos)){
                foreach($medicamentos_nuevos as $med) {
                    if (isset($med['producto_id']) && $med['producto_id'] == $producto_id) {
                        $encontrado = true;
                        break;
                    }
                }
            }
            if (!$encontrado) {
                $diferencias_stock[$producto_id] = ($diferencias_stock[$producto_id] ?? 0) + $cantidad_antigua;
            }
        }

        // --- 3. PROCESAR SUMINISTROS DE CONSUMO ---
        $suministros_usados = $_POST['suministros'] ?? [];
        $stmt_consumo_anterior = $conn->prepare("SELECT producto_id, cantidad_usada FROM jornada_consumo WHERE jornada_cliente_id = ?");
        $stmt_consumo_anterior->execute([$jornada_cliente_id]);
        $consumo_anterior = $stmt_consumo_anterior->fetchAll(PDO::FETCH_KEY_PAIR);
        $conn->prepare("DELETE FROM jornada_consumo WHERE jornada_cliente_id = ?")->execute([$jornada_cliente_id]);
        $stmt_insert_consumo = $conn->prepare("INSERT INTO jornada_consumo (jornada_cliente_id, producto_id, cantidad_usada) VALUES (?, ?, ?)");

        foreach ($suministros_usados as $producto_id => $cantidad_nueva) {
            $cantidad_nueva = (int)$cantidad_nueva;
            $cantidad_anterior = (int)($consumo_anterior[$producto_id] ?? 0);
            $diferencia = $cantidad_anterior - $cantidad_nueva;
            $diferencias_stock[$producto_id] = ($diferencias_stock[$producto_id] ?? 0) + $diferencia;
            if ($cantidad_nueva > 0) {
                $stmt_insert_consumo->execute([$jornada_cliente_id, $producto_id, $cantidad_nueva]);
            }
        }

        // --- 4. APLICAR TODAS LAS DIFERENCIAS DE STOCK ---
        $stmt_update_stock_ruta = $conn->prepare("UPDATE ruta_stock SET cantidad = cantidad + ? WHERE ruta_id = ? AND producto_id = ?");
        foreach ($diferencias_stock as $producto_id => $diferencia) {
            if ($diferencia != 0) {
                $stmt_update_stock_ruta->execute([$diferencia, $ruta_id, $producto_id]);
            }
        }
        
        // --- 5. ACTUALIZAR DATOS FINALES DEL PACIENTE ---
        $pago_divisas = floatval($_POST['pago_divisas'] ?? 0);
        $pago_bs_efectivo = floatval($_POST['pago_bs_efectivo'] ?? 0);
        $pago_pesos = floatval($_POST['pago_pesos'] ?? 0);
        $pago_zelle = floatval($_POST['pago_zelle'] ?? 0);
        $pago_punto = floatval($_POST['pago_punto'] ?? 0);
        $pago_movil = floatval($_POST['pago_movil'] ?? 0);
        
        $sql = "UPDATE jornada_clientes SET 
                    estado_cita = ?, notas_coordinador = ?, total_facturado = ?,
                    pago_divisas = ?, pago_bs_efectivo = ?, pago_pesos = ?, 
                    pago_zelle = ?, pago_punto = ?, pago_movil = ?
                ";
        $params = [
            $_POST['estado_cita'], $_POST['notas_coordinador'], $total_facturado,
            $pago_divisas, $pago_bs_efectivo, $pago_pesos,
            $pago_zelle, $pago_punto, $pago_movil
        ];
        
        if ($comprobante_filename) {
            $sql .= ", comprobante_pago = ?";
            $params[] = $comprobante_filename;
        } else {
             $sql .= ", metodo_pago = ?"; // Solo actualizar si no se sube un nuevo comprobante
             $params[] = $_POST['metodo_pago'] ?? null; // Evita undefined array key
        }
        
        $sql .= " WHERE id = ?";
        $params[] = $jornada_cliente_id;
        
        $stmt = $conn->prepare($sql);
        $stmt->execute($params);

        // === INICIO: Propagar estado a clientes y prospectos ===
        try {
            // Obtener cliente_id desde jornada_clientes
            $stmt_cliente_id = $conn->prepare("SELECT cliente_id FROM jornada_clientes WHERE id = ?");
            $stmt_cliente_id->execute([$jornada_cliente_id]);
            $cliente_id = $stmt_cliente_id->fetchColumn();

            if (!$cliente_id) {
                throw new Exception("No se encontró cliente asociado al registro de jornada.");
            }

            $nuevo_estado = $_POST['estado_cita'] ?? null;
            if (!$nuevo_estado) {
                throw new Exception("Estado de cita no proporcionado.");
            }

            error_log("🔍 DEBUG: Estado recibido para actualización: '{$nuevo_estado}'");

            // Obtener fecha_jornada
            $stmt_fecha = $conn->prepare("
                SELECT j.fecha_jornada 
                FROM jornadas j 
                JOIN jornada_clientes jc ON j.id = jc.jornada_id 
                WHERE jc.id = ?
            ");
            $stmt_fecha->execute([$jornada_cliente_id]);
            $fecha_jornada = $stmt_fecha->fetchColumn();
            $fecha_jornada_obj = new DateTime($fecha_jornada);

            // --- OBTENER CÉDULA DEL CLIENTE ---
            $stmt_cliente_data = $conn->prepare("SELECT cedula, nombre FROM clientes WHERE id = ?");
            $stmt_cliente_data->execute([$cliente_id]);
            $cliente_data = $stmt_cliente_data->fetch(PDO::FETCH_ASSOC);

            if (!$cliente_data || empty($cliente_data['cedula'])) {
                throw new Exception("Cliente ID {$cliente_id} no tiene cédula registrada.");
            }

            $cedula_cliente = $cliente_data['cedula'];
            error_log("🔍 DEBUG: Buscando prospecto con cédula: '{$cedula_cliente}'");

            // --- ACTUALIZAR TABLA CLIENTES ---
            $update_cliente_sql = "UPDATE clientes SET estado = ?";
            $update_cliente_params = [$nuevo_estado];

            $estado_lower = strtolower(trim($nuevo_estado));
            if (in_array($estado_lower, ['asistido', 'atendido', 'consultado', 'agendado'])) {
                $update_cliente_sql .= ", fecha_ultima_jornada_asistida = ?";
                $update_cliente_params[] = $fecha_jornada_obj->format('Y-m-d');
            }
            $update_cliente_sql .= " WHERE id = ?";
            $update_cliente_params[] = $cliente_id;

            $stmt_update_cliente = $conn->prepare($update_cliente_sql);
            $stmt_update_cliente->execute($update_cliente_params);

            error_log("✅ Cliente ID {$cliente_id} ({$cliente_data['nombre']}) actualizado a estado '{$nuevo_estado}'");

            // --- BUSCAR Y ACTUALIZAR PROSPECTO POR CÉDULA (usando columna `estado`) ---
            $stmt_check_prospecto = $conn->prepare("SELECT id, nombre FROM prospectos WHERE cedula = ?");
            $stmt_check_prospecto->execute([$cedula_cliente]);
            $prospecto = $stmt_check_prospecto->fetch(PDO::FETCH_ASSOC);

            if ($prospecto) {
                $stmt_update_prospecto = $conn->prepare("UPDATE prospectos SET estado = ? WHERE cedula = ?");
                $stmt_update_prospecto->execute([$nuevo_estado, $cedula_cliente]);

                error_log("✅ ÉXITO: Prospecto ID {$prospecto['id']} ({$prospecto['nombre']}) actualizado a estado '{$nuevo_estado}'");
            } else {
                error_log("❌ NO ENCONTRADO: No hay prospecto con cédula = '{$cedula_cliente}'");
            }

        } catch (Exception $e) {
            error_log("💥 Error al sincronizar estado con clientes/prospectos: " . $e->getMessage());
        }
        // === FIN: Propagación de estado ===

        $conn->commit();
        
        $accion_desc = $es_superadmin ? 
            "Superadmin actualizó datos, pagos y consumo para el paciente-jornada ID: {$jornada_cliente_id}" :
            "Actualizó datos, pagos y consumo para el paciente-jornada ID: {$jornada_cliente_id}";
        registrar_accion('actualiza_paciente_jornada', $accion_desc);
        
        $response = ['success' => true, 'message' => 'Información del paciente actualizada correctamente.'];

    } catch (Exception $e) {
        $conn->rollBack();
        $response['message'] = 'Error al actualizar: ' . $e->getMessage();
        error_log("Error en actualizar_paciente_jornada: " . $e->getMessage());
    }
    break;

    default:
        $response['message'] = 'Acción desconocida.';
        break;
}

echo json_encode($response);
?>