<?php
define('IS_AJAX', true);
require_once '../administracion/includes/seguridad.php';

header('Content-Type: application/json');
$response = ['success' => false, 'message' => 'Acción no válida o error inesperado.'];
$usuario_id = $_SESSION['user_id'] ?? 0;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        $response['message'] = 'Error de seguridad (CSRF). Por favor, recargue la página.';
        echo json_encode($response);
        exit();
    }
}

$action = $_REQUEST['action'] ?? '';

if (!puede('realizar_cuadre_caja')) {
    $response['message'] = 'Acceso denegado. No tiene permiso para realizar cuadres de caja.';
    echo json_encode($response);
    exit();
}

try {
    switch ($action) {
        case 'get_rutas_para_cuadre':
            $stmt = $conn->query("
                SELECT DISTINCT r.id, r.nombre_ruta
                FROM rutas r
                JOIN jornadas j ON r.id = j.ruta_id
                WHERE r.archivada = 0 AND j.id NOT IN (SELECT jornada_id FROM jornada_cuadres)
                ORDER BY r.nombre_ruta
            ");
            $rutas = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $response = ['success' => true, 'data' => $rutas];
            break;

        case 'get_jornadas_por_ruta':
            $ruta_id = filter_input(INPUT_GET, 'ruta_id', FILTER_VALIDATE_INT);
            if (!$ruta_id) { $response['message'] = 'ID de ruta no válido.'; break; }
            
            $stmt = $conn->prepare("
                SELECT j.id, j.ciudad_nombre, j.fecha_jornada
                FROM jornadas j
                LEFT JOIN jornada_cuadres jc ON j.id = jc.jornada_id
                WHERE j.ruta_id = ? AND jc.id IS NULL
                ORDER BY j.fecha_jornada DESC
            ");
            $stmt->execute([$ruta_id]);
            $jornadas = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $response = ['success' => true, 'data' => $jornadas];
            break;

        case 'get_datos_cuadre':
            $jornada_id = filter_input(INPUT_GET, 'jornada_id', FILTER_VALIDATE_INT);
            if (!$jornada_id) { $response['message'] = 'ID de jornada no válido.'; break; }

            $stmt_jornada = $conn->prepare("SELECT ruta_id, fecha_jornada FROM jornadas WHERE id = ?");
            $stmt_jornada->execute([$jornada_id]);
            $jornada_actual = $stmt_jornada->fetch(PDO::FETCH_ASSOC);
            if (!$jornada_actual) { $response['message'] = 'Jornada no encontrada.'; break; }

            $stmt_saldo = $conn->prepare("SELECT jc.neto_saldo_final_usd FROM jornada_cuadres jc JOIN jornadas j ON jc.jornada_id = j.id WHERE j.ruta_id = ? AND j.fecha_jornada < ? ORDER BY j.fecha_jornada DESC LIMIT 1");
            $stmt_saldo->execute([$jornada_actual['ruta_id'], $jornada_actual['fecha_jornada']]);
            $saldo_anterior = $stmt_saldo->fetchColumn() ?: 0.00;

            $stmt_ingresos = $conn->prepare("SELECT COALESCE(SUM(pago_divisas), 0) as total_divisas, COALESCE(SUM(pago_bs_efectivo), 0) as total_bs_efectivo, COALESCE(SUM(pago_pesos), 0) as total_pesos, COALESCE(SUM(pago_zelle), 0) as total_zelle, COALESCE(SUM(pago_punto), 0) as total_punto, COALESCE(SUM(pago_movil), 0) as total_movil FROM jornada_clientes WHERE jornada_id = ?");
            $stmt_ingresos->execute([$jornada_id]);
            $ingresos = $stmt_ingresos->fetch(PDO::FETCH_ASSOC);
            
            $stmt_medicamentos_total = $conn->prepare("SELECT COALESCE(SUM(jm.cantidad * jm.precio), 0) as total FROM jornada_medicamentos jm JOIN jornada_clientes jc ON jm.jornada_cliente_id = jc.id WHERE jc.jornada_id = ?");
            $stmt_medicamentos_total->execute([$jornada_id]);
            $total_medicamentos_usd = $stmt_medicamentos_total->fetchColumn();

            $stmt_gastos = $conn->prepare("SELECT * FROM jornada_gastos WHERE jornada_id = ? ORDER BY fecha_gasto DESC");
            $stmt_gastos->execute([$jornada_id]);
            $gastos = $stmt_gastos->fetchAll(PDO::FETCH_ASSOC);

            $stmt_liquidaciones = $conn->prepare("SELECT * FROM jornada_liquidaciones WHERE jornada_id = ? ORDER BY fecha_liquidacion DESC");
            $stmt_liquidaciones->execute([$jornada_id]);
            $liquidaciones = $stmt_liquidaciones->fetchAll(PDO::FETCH_ASSOC);

            $stmt_pacientes = $conn->prepare("SELECT jc.id as jornada_cliente_id, c.nombre as nombre_cliente, c.cedula as cedula_cliente, jc.total_facturado, jc.pago_divisas, jc.pago_bs_efectivo, jc.pago_pesos, jc.pago_zelle, jc.pago_punto, jc.pago_movil FROM jornada_clientes jc JOIN clientes c ON jc.cliente_id = c.id WHERE jc.jornada_id = ? AND jc.estado_cita != 'Cancelado'");
            $stmt_pacientes->execute([$jornada_id]);
            $pacientes = $stmt_pacientes->fetchAll(PDO::FETCH_ASSOC);
            
            foreach ($pacientes as $key => $paciente) {
                $stmt_medicamentos = $conn->prepare("SELECT cantidad, nombre_medicamento, precio FROM jornada_medicamentos WHERE jornada_cliente_id = ?");
                $stmt_medicamentos->execute([$paciente['jornada_cliente_id']]);
                $pacientes[$key]['medicamentos'] = $stmt_medicamentos->fetchAll(PDO::FETCH_ASSOC);
            }

            $response = ['success' => true, 'data' => [
                'saldo_brigada_anterior' => $saldo_anterior,
                'ingresos' => $ingresos,
                'total_medicamentos_usd' => $total_medicamentos_usd,
                'gastos' => $gastos,
                'liquidaciones' => $liquidaciones,
                'pacientes' => $pacientes
            ]];
            break;

        case 'guardar_gasto':
            $jornada_id = filter_input(INPUT_POST, 'jornada_id', FILTER_VALIDATE_INT);
            $gasto_id = filter_input(INPUT_POST, 'gasto_id', FILTER_VALIDATE_INT);
            $concepto = trim($_POST['concepto'] ?? '');
            $monto = floatval($_POST['monto'] ?? 0);
            $moneda = $_POST['moneda'] ?? 'USD';
            $descripcion = trim($_POST['descripcion'] ?? '');
            if (!$jornada_id || empty($concepto) || $monto <= 0) { $response['message'] = 'Todos los campos del gasto son obligatorios.'; break; }
            if ($gasto_id) {
                $stmt = $conn->prepare("UPDATE jornada_gastos SET concepto=?, monto=?, moneda=?, descripcion=? WHERE id=?");
                $stmt->execute([$concepto, $monto, $moneda, $descripcion, $gasto_id]);
                $response = ['success' => true, 'message' => 'Gasto actualizado.'];
            } else {
                $stmt = $conn->prepare("INSERT INTO jornada_gastos (jornada_id, concepto, monto, moneda, descripcion, registrado_por_id) VALUES (?, ?, ?, ?, ?, ?)");
                $stmt->execute([$jornada_id, $concepto, $monto, $moneda, $descripcion, $usuario_id]);
                $response = ['success' => true, 'message' => 'Gasto registrado.'];
            }
            break;

        case 'eliminar_gasto':
            $gasto_id = filter_input(INPUT_POST, 'gasto_id', FILTER_VALIDATE_INT);
            if (!$gasto_id) { $response['message'] = 'ID de gasto no válido.'; break; }
            $stmt = $conn->prepare("DELETE FROM jornada_gastos WHERE id = ?");
            $stmt->execute([$gasto_id]);
            $response = ['success' => true, 'message' => 'Gasto eliminado.'];
            break;

        case 'guardar_liquidacion':
            $jornada_id = filter_input(INPUT_POST, 'jornada_id', FILTER_VALIDATE_INT);
            $liquidacion_id = filter_input(INPUT_POST, 'liquidacion_id', FILTER_VALIDATE_INT);
            $metodo = trim($_POST['metodo_pago'] ?? '');
            $monto = floatval($_POST['monto_usd'] ?? 0);
            $referencia = trim($_POST['referencia'] ?? '');
            if (!$jornada_id || empty($metodo) || $monto <= 0) { $response['message'] = 'Método y monto son obligatorios para la liquidación.'; break; }
            if ($liquidacion_id) {
                $stmt = $conn->prepare("UPDATE jornada_liquidaciones SET metodo_pago=?, monto_usd=?, referencia=? WHERE id=?");
                $stmt->execute([$metodo, $monto, $referencia, $liquidacion_id]);
                $response = ['success' => true, 'message' => 'Liquidación actualizada.'];
            } else {
                $stmt = $conn->prepare("INSERT INTO jornada_liquidaciones (jornada_id, metodo_pago, monto_usd, referencia, registrado_por_id) VALUES (?, ?, ?, ?, ?)");
                $stmt->execute([$jornada_id, $metodo, $monto, $referencia, $usuario_id]);
                $response = ['success' => true, 'message' => 'Liquidación registrada.'];
            }
            break;
            
        case 'eliminar_liquidacion':
            $liquidacion_id = filter_input(INPUT_POST, 'liquidacion_id', FILTER_VALIDATE_INT);
            if (!$liquidacion_id) { $response['message'] = 'ID de liquidación no válido.'; break; }
            $stmt = $conn->prepare("DELETE FROM jornada_liquidaciones WHERE id = ?");
            $stmt->execute([$liquidacion_id]);
            $response = ['success' => true, 'message' => 'Liquidación eliminada.'];
            break;

        case 'guardar_cuadre_final':
            $cuadre = json_decode($_POST['cuadre_data'] ?? '{}', true);
            if (json_last_error() !== JSON_ERROR_NONE || empty($cuadre)) { $response['message'] = 'Datos del cuadre inválidos.'; break; }

            $conn->beginTransaction();
            $stmt = $conn->prepare("INSERT INTO jornada_cuadres (jornada_id, admin_id, tasa_bs_usd, tasa_cop_usd, base_inicial_usd, saldo_brigada_anterior_usd, total_ingresos_usd, total_medicamentos_usd, total_gastos_usd, total_liquidado_usd, neto_saldo_final_usd, notas) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([
                $cuadre['jornada_id'], $usuario_id, $cuadre['tasa_bs'], $cuadre['tasa_cop'], 
                $cuadre['base_inicial'], $cuadre['saldo_brigada_anterior'],
                $cuadre['total_ingresos_usd'], $cuadre['total_medicamentos_usd'], $cuadre['total_gastos_usd'],
                $cuadre['total_liquidado_usd'], $cuadre['neto_saldo_final_usd'],
                $cuadre['notas']
            ]);
            $stmtUpdate = $conn->prepare("UPDATE jornadas SET estado_jornada = 'Completada' WHERE id = ?");
            $stmtUpdate->execute([$cuadre['jornada_id']]);
            $conn->commit();
            $response = ['success' => true, 'message' => '¡Cuadre guardado y jornada cerrada!'];
            break;
            
        case 'get_historial_cuadres':
            $stmt = $conn->query("SELECT jc.*, r.nombre_ruta, j.ciudad_nombre, u.nombre as admin_nombre FROM jornada_cuadres jc JOIN jornadas j ON jc.jornada_id = j.id JOIN rutas r ON j.ruta_id = r.id JOIN usuarios u ON jc.admin_id = u.id ORDER BY jc.fecha_cuadre DESC");
            $historial = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $response = ['success' => true, 'data' => $historial];
            break;

        default:
            $response['message'] = 'Acción de cuadre desconocida.';
            break;
    }
} catch (PDOException $e) {
    if ($conn->inTransaction()) { $conn->rollBack(); }
    error_log('Error en cuadre_ajax.php: ' . $e->getMessage());
    $response['message'] = 'Ocurrió un error en el servidor. Por favor, contacte al administrador.';
}

echo json_encode($response);
?>