<?php
define('IS_AJAX', true);
require_once '../administracion/includes/seguridad.php';

header('Content-Type: application/json');
$response = ['success' => false, 'message' => 'Acción no válida o sin permisos.'];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        $response['message'] = 'Error de seguridad (CSRF).';
        echo json_encode($response);
        exit();
    }
}

$action = $_POST['action'] ?? $_GET['action'] ?? '';

switch ($action) {
    case 'cargar_datos_iniciales':
        if (!puede('gestionar_nomina')) break;
        try {
            // Consulta MEJORADA: une usuarios con su configuración de nómina
            $stmt = $conn->prepare("
                SELECT 
                    u.id, u.nombre, u.cedula, u.salario_base, r.nombre as cargo,
                    IFNULL(nc.cesta_ticket, 40.00) as cesta_ticket,
                    IFNULL(nc.complemento_cesta_ticket, 30.00) as complemento_cesta_ticket,
                    IFNULL(nc.bono_transporte, 40.00) as bono_transporte,
                    IFNULL(nc.complemento_productividad, 0.00) as complemento_productividad
                FROM usuarios u 
                LEFT JOIN roles r ON u.rol_id = r.id 
                LEFT JOIN nomina_configuracion nc ON u.id = nc.usuario_id
                WHERE u.estado = 1 AND u.rol_id != 99999999999999 
                ORDER BY u.nombre ASC
            ");
            $stmt->execute();
            $empleados = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $response = ['success' => true, 'data' => ['empleados' => $empleados]];
        } catch (PDOException $e) {
            error_log("Nominas AJAX (cargar_datos_iniciales): " . $e->getMessage());
            $response['message'] = 'Error al cargar los datos de los empleados.';
        }
        break;

    case 'crear_nomina':
        if (!puede('gestionar_nomina')) break;
        
        $nombre_nomina = trim($_POST['nombre_nomina'] ?? '');
        $fecha_inicio = trim($_POST['fecha_inicio'] ?? '');
        $fecha_fin = trim($_POST['fecha_fin'] ?? '');
        $tasa_cambio = floatval($_POST['tasa_cambio'] ?? 0);
        $detalles = json_decode($_POST['detalles'] ?? '[]', true);

        if (empty($nombre_nomina) || empty($fecha_inicio) || empty($fecha_fin) || $tasa_cambio <= 0 || empty($detalles)) {
            $response['message'] = 'Faltan datos. Verifique el nombre, las fechas, la tasa de cambio y que haya añadido empleados.';
            break;
        }

        try {
            $conn->beginTransaction();
            $stmt_nomina = $conn->prepare("INSERT INTO nominas (nombre_nomina, fecha_inicio, fecha_fin, tasa_cambio, creado_por_id, estado) VALUES (?, ?, ?, ?, ?, 'Generada')");
            $stmt_nomina->execute([$nombre_nomina, $fecha_inicio, $fecha_fin, $tasa_cambio, $_SESSION['user_id']]);
            $nomina_id = $conn->lastInsertId();

            $stmt_detalle = $conn->prepare(
                "INSERT INTO nomina_detalles (
                    nomina_id, empleado_id, sueldo_base, salario_diario_usd, cesta_ticket, cesta_ticket_diario_usd, 
                    complemento_cesta_ticket, comp_cesta_ticket_diario_usd, complemento_productividad, bono_transporte, 
                    bono_transporte_diario_usd, total_asignaciones, adelantos, dias_no_laborados, deducciones, total_deducciones, 
                    total_pagar, total_pagar_bs, observacion
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)"
            );

            foreach ($detalles as $d) {
                $stmt_detalle->execute([
                    $nomina_id, $d['empleado_id'], $d['sueldo_base'], $d['salario_diario_usd'], $d['cesta_ticket'], 
                    $d['cesta_ticket_diario_usd'], $d['complemento_cesta_ticket'], $d['comp_cesta_ticket_diario_usd'], 
                    $d['complemento_productividad'], $d['bono_transporte'], $d['bono_transporte_diario_usd'], 
                    $d['total_asignaciones'], $d['adelantos'], $d['dias_no_laborados'], $d['deducciones'], $d['total_deducciones'], 
                    $d['total_pagar'], $d['total_pagar_bs'], trim($d['observacion'] ?? '')
                ]);
            }
            $conn->commit();
            registrar_accion('creacion_nomina', "Creó la nómina '{$nombre_nomina}' (ID: {$nomina_id}) con " . count($detalles) . " empleados.");
            $response = ['success' => true, 'message' => 'Nómina creada exitosamente.'];
        } catch (PDOException $e) {
            $conn->rollBack();
            error_log("Nominas AJAX (crear_nomina): " . $e->getMessage());
            // ----- AQUÍ ESTÁ LA CORRECCIÓN -----
            // Ahora te mostrará el error específico de la base de datos.
            $response['message'] = 'Error de base de datos: ' . $e->getMessage();
        }
        break;
        
    case 'obtener_detalle_nomina':
        if (!puede('ver_nomina') && !puede('gestionar_nomina')) break;

        $nomina_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
        if (!$nomina_id) {
            $response['message'] = 'ID de nómina no válido.';
            break;
        }

        try {
            // Consulta para los datos generales de la nómina
            $stmt_nomina = $conn->prepare("
                SELECT n.*, u.nombre as creado_por
                FROM nominas n
                JOIN usuarios u ON n.creado_por_id = u.id
                WHERE n.id = ?
            ");
            $stmt_nomina->execute([$nomina_id]);
            $nomina = $stmt_nomina->fetch(PDO::FETCH_ASSOC);

            if (!$nomina) {
                $response['message'] = 'No se encontró la nómina solicitada.';
                break;
            }

            // Consulta para los detalles de los empleados en esa nómina
            $stmt_detalles = $conn->prepare("
                SELECT nd.*, u.nombre as nombre_empleado, u.cedula
                FROM nomina_detalles nd
                JOIN usuarios u ON nd.empleado_id = u.id
                WHERE nd.nomina_id = ?
                ORDER BY u.nombre ASC
            ");
            $stmt_detalles->execute([$nomina_id]);
            $detalles = $stmt_detalles->fetchAll(PDO::FETCH_ASSOC);

            $response = [
                'success' => true,
                'data' => [
                    'nomina' => $nomina,
                    'detalles' => $detalles
                ]
            ];

        } catch (PDOException $e) {
            error_log("Nominas AJAX (obtener_detalle_nomina): " . $e->getMessage());
            $response['message'] = 'Error de base de datos al obtener los detalles.';
        }
        break;

    case 'marcar_como_pagada':
        // Solo quien puede gestionar puede marcar como pagada
        if (!puede('gestionar_nomina')) break;

        $nomina_id = filter_input(INPUT_POST, 'nomina_id', FILTER_VALIDATE_INT);
        if (!$nomina_id) {
            $response['message'] = 'ID de nómina no válido.';
            break;
        }

        try {
            $stmt = $conn->prepare("UPDATE nominas SET estado = 'Pagada' WHERE id = ? AND estado = 'Generada'");
            $stmt->execute([$nomina_id]);

            if ($stmt->rowCount() > 0) {
                registrar_accion('nomina_pagada', "Marcó la nómina ID {$nomina_id} como Pagada.");
                $response = ['success' => true, 'message' => 'Nómina actualizada a "Pagada" exitosamente.'];
            } else {
                $response['message'] = 'La nómina no se pudo actualizar. Es posible que ya estuviera pagada o no exista.';
            }

        } catch (PDOException $e) {
            error_log("Nominas AJAX (marcar_como_pagada): " . $e->getMessage());
            $response['message'] = 'Error de base de datos: ' . $e->getMessage();
        }
        break;    

    case 'listar_nominas':
        if (!puede('ver_nomina') && !puede('gestionar_nomina')) break;
        try {
            $stmt = $conn->prepare("SELECT n.id, n.nombre_nomina, n.fecha_inicio, n.fecha_fin, n.estado, n.fecha_creacion, u.nombre as creado_por FROM nominas n JOIN usuarios u ON n.creado_por_id = u.id ORDER BY n.fecha_creacion DESC");
            $stmt->execute();
            $nominas = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $response = ['success' => true, 'data' => $nominas];
        } catch (PDOException $e) {
            error_log("Nominas AJAX (listar_nominas): " . $e->getMessage());
            $response['message'] = 'Error al obtener el listado de nóminas.';
        }
        break;
}

echo json_encode($response);