<?php
// Archivo: /api/pdf_almacen_ajax.php (NUEVO)

require_once '../administracion/includes/seguridad.php';
require_once 'lib/fpdf.php'; // Asegúrate de que la ruta a fpdf.php sea correcta

/**
 * Clase personalizada para crear el PDF con un encabezado que incluye logo y fondo.
 */
class PDF extends FPDF {
    // Cabecera de página
    function Header() {
        // Imagen de fondo (se coloca primero para que quede detrás de todo)
        // La imagen debe tener un ancho apropiado para una hoja tamaño carta (216 mm)
        if (file_exists('../assets/images/fondo.png')) {
            $this->Image('../assets/images/fondo.png', 0, 0, 216);
        }

        // Logo de la empresa (posicionado en la esquina superior izquierda)
        if (file_exists('../assets/images/logo.png')) {
            $this->Image('../assets/images/logo.png', 10, 8, 33); // Eje X, Eje Y, Ancho
        }

        // Salto de línea para empezar el contenido debajo del logo
        $this->Ln(20);
    }
}

// Verificación de permisos y del ID del despacho
if (!isset($_GET['id']) || !puede('ver_inventario')) {
    die('Acceso denegado o ID de despacho no proporcionado.');
}

$despacho_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
if (!$despacho_id) {
    die('ID de despacho no válido.');
}

try {
    // 1. Obtener los datos principales del despacho desde la base de datos
    $stmt = $conn->prepare("
        SELECT 
            d.fecha_despacho,
            r.nombre_ruta,
            u_despacha.nombre as nombre_despacha,
            u_coord.nombre as nombre_coord
        FROM almacen_despachos d
        JOIN rutas r ON d.ruta_id = r.id
        LEFT JOIN usuarios u_despacha ON d.usuario_id = u_despacha.id
        LEFT JOIN usuarios u_coord ON r.coordinador_id = u_coord.id
        WHERE d.id = ?
    ");
    $stmt->execute([$despacho_id]);
    $despacho = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$despacho) {
        die('Despacho no encontrado.');
    }

    // 2. Obtener la lista de productos (items) de ese despacho
    $stmt_items = $conn->prepare("
        SELECT p.nombre, di.cantidad_enviada
        FROM almacen_despacho_items di
        JOIN almacen_productos p ON di.producto_id = p.id
        WHERE di.despacho_id = ? ORDER BY p.nombre ASC
    ");
    $stmt_items->execute([$despacho_id]);
    $items = $stmt_items->fetchAll(PDO::FETCH_ASSOC);

    // --- INICIO DE LA CONSTRUCCIÓN DEL PDF ---

    $pdf = new PDF('P', 'mm', 'Letter'); // Usamos nuestra clase personalizada
    $pdf->AddPage();
    $pdf->SetFont('Arial', '', 12);

    // Fecha (posicionada debajo del logo)
    $pdf->Cell(0, 10, utf8_decode('Mérida, ') . date('d \d\e F \d\e Y', strtotime($despacho['fecha_despacho'])), 0, 1, 'R');
    $pdf->Ln(10);

    // NOTA: Aquí está la lógica del coordinador que mencionaste.
    // Por ahora, toma el coordinador asignado a la ruta.
    $nombre_coordinador = $despacho['nombre_coord'] ?? 'Coordinador(a) de Ruta';

    // Bloque del Destinatario
    $pdf->SetFont('Arial', 'B', 12);
    $pdf->Cell(0, 7, utf8_decode('Sra. ' . $nombre_coordinador), 0, 1);
    $pdf->SetFont('Arial', '', 12);
    $pdf->Cell(0, 7, utf8_decode('Coordinadora AMI ' . $despacho['nombre_ruta']), 0, 1);
    $pdf->Cell(0, 7, 'Presente.-', 0, 1);
    $pdf->Ln(10);
    
    // Bloque del Asunto
    $pdf->SetFont('Arial', 'B', 12);
    $pdf->Cell(20, 7, 'Asunto: ', 0, 0);
    $pdf->SetFont('Arial', '', 12);
    $pdf->Cell(0, 7, utf8_decode('Despacho de medicamentos e insumos'), 0, 1);
    $pdf->Ln(5);

    // Cuerpo del mensaje
    $nombre_despacha = $despacho['nombre_despacha'] ?? 'El Despacho';
    $pdf->MultiCell(0, 7, utf8_decode("Quien suscribe, {$nombre_despacha} Jefe de Despacho, por medio de la presente le hago entrega de los siguientes medicamentos y pruebas:"), 0, 'J');
    $pdf->Ln(10);

    // Tabla de productos
    $pdf->SetFont('Arial', 'B', 11);
    $pdf->SetFillColor(230, 230, 230); // Un gris claro para el encabezado de la tabla
    $pdf->Cell(150, 8, 'PRODUCTOS', 1, 0, 'C', true);
    $pdf->Cell(40, 8, 'CANTIDAD', 1, 1, 'C', true);

    $pdf->SetFont('Arial', '', 10);
    $total_unidades = 0;
    foreach ($items as $item) {
        $pdf->Cell(150, 8, utf8_decode($item['nombre']), 1, 0, 'L');
        $pdf->Cell(40, 8, $item['cantidad_enviada'], 1, 1, 'C');
        $total_unidades += $item['cantidad_enviada'];
    }
    $pdf->Ln(5);
    
    // Total de unidades
    $pdf->SetFont('Arial', 'B', 11);
    $pdf->Cell(150, 8, 'TOTAL DE UNIDADES', 1, 0, 'R');
    $pdf->Cell(40, 8, $total_unidades, 1, 1, 'C');
    $pdf->Ln(25);

    // Bloque de Firmas
    $pdf->SetFont('Arial', '', 12);
    $pdf->Cell(95, 7, 'Entrega', 0, 0, 'C');
    $pdf->Cell(95, 7, 'Recibe', 0, 1, 'C');
    $pdf->Ln(15);
    $pdf->Cell(95, 7, '____________________', 0, 0, 'C');
    $pdf->Cell(95, 7, '____________________', 0, 1, 'C');
    $pdf->Cell(95, 7, utf8_decode($nombre_despacha), 0, 0, 'C');
    $pdf->Cell(95, 7, utf8_decode($nombre_coordinador), 0, 1, 'C');

    // Envía el PDF al navegador para descarga
    $pdf->Output('D', 'despacho_'.$despacho_id.'.pdf');
    exit();

} catch (Exception $e) {
    // En caso de error, muestra un mensaje simple
    die('Error al generar el PDF: ' . $e->getMessage());
}
?>