<?php
define('IS_AJAX', true);
require_once '../administracion/includes/seguridad.php';

header('Content-Type: application/json');
$response = ['success' => false, 'message' => 'Acción no válida.'];
$usuario_id = $_SESSION['user_id'] ?? 0;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        $response['message'] = 'Error de seguridad (CSRF).';
        echo json_encode($response);
        exit();
    }
}

$action = $_POST['action'] ?? $_GET['action'] ?? '';

// --- ACCIONES DE ADMINISTRADOR Y SUPERVISOR (Se mantienen) ---
if (in_array($action, ['cargar_dashboard_admin', 'crear_lote_y_asignar', 'get_detalles_lote'])) {
    if (!puede('gestionar_prospectos_admin')) {
        $response['message'] = 'Acceso denegado.';
        echo json_encode($response);
        exit();
    }
    // El código para las acciones de administrador se puede añadir aquí si es necesario
    
}
// --- ACCIONES PARA OPERADOR/ASESOR ---
else if (puede('ver_prospectos_operador')) {
    switch ($action) {
        case 'cargar_mis_prospectos':
    try {
        // --- INICIO DE LA LÓGICA CORREGIDA Y ROBUSTA ---

        // 1. Obtener prospectos normales (AHORA INCLUYE ORIGEN Y ORIGEN_OTRO)
        $stmt_prospectos = $conn->prepare("
            SELECT 
                id, NULL as reactivacion_id, 'Prospecto' as tipo, nombre, cedula, telefono, ciudad, campana, 
                origen, origen_otro, -- CAMPOS NUEVOS
                estado, ultima_accion, fecha_creacion, notas
            FROM prospectos 
            WHERE operador_id = ?
        ");
        $stmt_prospectos->execute([$usuario_id]);
        $prospectos = $stmt_prospectos->fetchAll(PDO::FETCH_ASSOC);

        // 2. Obtener clientes de reactivación
        $stmt_reactivacion = $conn->prepare("
            SELECT 
                c.id, rc.id as reactivacion_id, 'Reactivación' as tipo, c.nombre, c.cedula, c.telefono, c.ciudad,
                rl.nombre_lote as campana, rc.estado_contacto as estado, '' as ultima_accion, 
                rc.fecha_asignacion_operador as fecha_creacion, rc.notas
            FROM reactivacion_clientes rc
            JOIN clientes c ON rc.cliente_id = c.id
            JOIN reactivacion_lotes rl ON rc.lote_id = rl.id
            WHERE rc.operador_id = ?
        ");
        $stmt_reactivacion->execute([$usuario_id]);
        $clientes_reactivacion = $stmt_reactivacion->fetchAll(PDO::FETCH_ASSOC);

        // 3. Unir los dos arrays en PHP (esto evita el error de SQL si uno está vacío)
        $todos_los_leads = array_merge($prospectos, $clientes_reactivacion);
        
        // 4. Aplicar filtros en PHP
        $search_term = strtolower($_GET['search'] ?? '');
        $filtro_campana = $_GET['campana'] ?? '';
        $filtro_fecha_inicio = $_GET['fecha_inicio'] ?? '';
        $filtro_fecha_fin = $_GET['fecha_fin'] ?? '';

        if (!empty($search_term) || !empty($filtro_campana) || !empty($filtro_fecha_inicio) || !empty($filtro_fecha_fin)) {
            $todos_los_leads = array_filter($todos_los_leads, function($lead) use ($search_term, $filtro_campana, $filtro_fecha_inicio, $filtro_fecha_fin) {
                $cumple_search = empty($search_term) || 
                                 strpos(strtolower($lead['nombre']), $search_term) !== false ||
                                 strpos(strtolower($lead['telefono']), $search_term) !== false ||
                                 strpos(strtolower($lead['ciudad']), $search_term) !== false;

                $cumple_campana = empty($filtro_campana) || $lead['campana'] == $filtro_campana;
                
                $fecha_lead = date('Y-m-d', strtotime($lead['fecha_creacion']));
                $cumple_inicio = empty($filtro_fecha_inicio) || $fecha_lead >= $filtro_fecha_inicio;
                $cumple_fin = empty($filtro_fecha_fin) || $fecha_lead <= $filtro_fecha_fin;

                return $cumple_search && $cumple_campana && $cumple_inicio && $cumple_fin;
            });
        }
        
        // 5. Ordenar el array combinado por fecha
        usort($todos_los_leads, function($a, $b) {
            return strtotime($b['fecha_creacion']) - strtotime($a['fecha_creacion']);
        });

        // 6. Lógica de paginación sobre el array final
        $total_records = count($todos_los_leads);
        $records_per_page = 5;
        $total_pages = $total_records > 0 ? ceil($total_records / $records_per_page) : 0;
        $current_page = filter_input(INPUT_GET, 'page', FILTER_VALIDATE_INT, ['options' => ['default' => 1, 'min_range' => 1]]);
        if ($current_page > $total_pages && $total_pages > 0) { $current_page = $total_pages; }
        $offset = ($current_page - 1) * $records_per_page;
        
        $leads_paginados = array_slice($todos_los_leads, $offset, $records_per_page);
        
        // Contar prospectos pendientes (esto se mantiene igual)
        $stmt_pendientes = $conn->prepare("SELECT COUNT(*) FROM prospectos WHERE operador_id = ? AND estado = 'nuevo'");
        $stmt_pendientes->execute([$usuario_id]);
        $pendientes_count = $stmt_pendientes->fetchColumn();
        
        // --- FIN DE LA LÓGICA CORREGIDA ---
        
        $response = [
            'success' => true, 'data' => $leads_paginados,
            'pendientes' => (int)$pendientes_count,
            'pagination' => ['total_records' => (int)$total_records, 'total_pages' => (int)$total_pages, 'current_page' => (int)$current_page]
        ];
    } catch (PDOException $e) {
        error_log("Error en cargar_mis_prospectos: " . $e->getMessage());
        $response['message'] = 'Error de base de datos al cargar tus prospectos.';
    }
    break;
        case 'agendar_cliente_reactivacion': // <-- INICIO DE LA NUEVA LÓGICA
            $cliente_id = filter_input(INPUT_POST, 'cliente_id', FILTER_VALIDATE_INT);
            $reactivacion_id = filter_input(INPUT_POST, 'reactivacion_id', FILTER_VALIDATE_INT);
            $jornada_id = filter_input(INPUT_POST, 'jornada_id', FILTER_VALIDATE_INT);
            $cedula = trim($_POST['cedula'] ?? '');
            $edad = filter_input(INPUT_POST, 'edad', FILTER_VALIDATE_INT);
            $sexo = in_array($_POST['sexo'] ?? '', ['M', 'F']) ? $_POST['sexo'] : null;
            $hora_asistencia = $_POST['hora_asistencia'] ?? null;
            $notas_agendamiento = trim($_POST['notas'] ?? '');

            if (!$cliente_id || !$reactivacion_id || !$jornada_id || empty($cedula) || !$edad || !$hora_asistencia || !$sexo) {
                $response['message'] = 'Faltan datos para agendar al cliente (Cédula, Edad, Sexo, Jornada, Hora).';
                break;
            }
            try {
                $conn->beginTransaction();

                // 1. Verificar que el cliente existe y pertenece al operador
                $stmt_cliente = $conn->prepare("
                    SELECT c.* FROM clientes c
                    JOIN reactivacion_clientes rc ON c.id = rc.cliente_id
                    WHERE c.id = ? AND rc.id = ? AND rc.operador_id = ?
                ");
                $stmt_cliente->execute([$cliente_id, $reactivacion_id, $usuario_id]);
                $cliente = $stmt_cliente->fetch(PDO::FETCH_ASSOC);

                if (!$cliente) {
                    $conn->rollBack();
                    $response['message'] = 'Cliente no encontrado o no tienes permiso sobre él.';
                    break;
                }

                // 2. Actualizar datos del cliente si es necesario (incluir sexo si viene en los datos)
                $sexo_cliente = isset($_POST['sexo']) && in_array($_POST['sexo'], ['M', 'F']) ? $_POST['sexo'] : null;
                $stmt_update_cliente = $conn->prepare("UPDATE clientes SET cedula = ?, edad = ?, sexo = COALESCE(?, sexo), estado = ?, estado_global = ?, fecha_ultima_actualizacion = NOW() WHERE id = ?");
                $stmt_update_cliente->execute([$cedula, $edad, $sexo_cliente, 'agendado', 'Agendado desde reactivación', $cliente_id]);

                // 3. Verificar que no esté ya agendado en esa jornada
                $stmt_check_agenda = $conn->prepare("SELECT id FROM jornada_clientes WHERE jornada_id = ? AND cliente_id = ?");
                $stmt_check_agenda->execute([$jornada_id, $cliente_id]);
                if ($stmt_check_agenda->fetch()) {
                    $conn->rollBack();
                    $response['message'] = 'Este cliente ya está agendado en la jornada seleccionada.';
                    break;
                }

                // 4. Agendar al cliente en la jornada
                $stmt_agendar = $conn->prepare("INSERT INTO jornada_clientes (jornada_id, cliente_id, agendado_por_operador_id, hora_asistencia, edad, sexo, estado_cita) VALUES (?, ?, ?, ?, ?, ?, 'Agendado')");
                $stmt_agendar->execute([$jornada_id, $cliente_id, $usuario_id, $hora_asistencia, $edad, $sexo]);

                // 5. Actualizar el estado en la tabla de reactivación
                $stmt_update_reactivacion = $conn->prepare("UPDATE reactivacion_clientes SET estado_contacto = 'Agendado' WHERE id = ?");
                $stmt_update_reactivacion->execute([$reactivacion_id]);
                
                $conn->commit();
                registrar_accion('agendamiento_reactivacion', "El operador agendó al cliente ID {$cliente_id} (Reactivación ID {$reactivacion_id}) a la jornada ID {$jornada_id}.");
                $response = ['success' => true, 'message' => '¡Cliente de reactivación agendado correctamente!'];

            } catch (Exception $e) {
                $conn->rollBack();
                $response['message'] = 'Error en el proceso de agendamiento: ' . $e->getMessage();
                error_log("Error en agendar_cliente_reactivacion: " . $e->getMessage());
            }
            break; // <-- FIN DE LA NUEVA LÓGICA
        
        case 'actualizar_cliente_reactivacion':
            $reactivacion_id = filter_input(INPUT_POST, 'reactivacion_id', FILTER_VALIDATE_INT);
            $estado_nuevo = $_POST['estado'] ?? '';
            $nota_adicional = trim($_POST['notas'] ?? '');

            if (!$reactivacion_id || empty($estado_nuevo)) {
                $response['message'] = 'Faltan datos para actualizar el cliente.';
                break;
            }

            try {
                $conn->beginTransaction();

                $stmt_check = $conn->prepare("SELECT notas FROM reactivacion_clientes WHERE id = ? AND operador_id = ?");
                $stmt_check->execute([$reactivacion_id, $usuario_id]);
                $cliente_actual = $stmt_check->fetch(PDO::FETCH_ASSOC);

                if (!$cliente_actual) {
                    $conn->rollBack();
                    $response['message'] = 'No tienes permiso para modificar este cliente.';
                    break;
                }

                $notas_para_guardar = $cliente_actual['notas'];
                if (!empty($nota_adicional)) {
                    $notas_para_guardar .= "\n---\n[" . date('Y-m-d H:i:s') . "]\n" . $nota_adicional;
                }

                $stmt_update = $conn->prepare("UPDATE reactivacion_clientes SET estado_contacto = ?, notas = ? WHERE id = ?");
                $stmt_update->execute([$estado_nuevo, $notas_para_guardar, $reactivacion_id]);
                
                $conn->commit();
                registrar_accion('actualiza_cliente_reactivacion', "Actualizó estado del cliente de reactivación ID: {$reactivacion_id} a '{$estado_nuevo}'");
                $response = ['success' => true, 'message' => 'Cliente de reactivación actualizado.'];

            } catch (PDOException $e) {
                $conn->rollBack();
                $response['message'] = 'Error de base de datos al actualizar.';
                error_log($e->getMessage());
            }
            break;    

        case 'get_jornadas_asignadas_operador':
            try {
                // --- CAMBIO PRINCIPAL AQUÍ ---
                // Se une la tabla `usuarios` con `roles` para obtener la 'clave' del rol
                $stmt_check_admin = $conn->prepare("
                    SELECT r.clave 
                    FROM usuarios u
                    JOIN roles r ON u.rol_id = r.id
                    WHERE u.id = ?
                ");
                $stmt_check_admin->execute([$usuario_id]);
                $usuario_rol_clave = $stmt_check_admin->fetchColumn();
                
                // Comparamos con la 'clave' de la tabla roles, que es 'superadmin'
                if (strtolower($usuario_rol_clave) === 'superadmin') {
                    // Si es superadministrador, mostrar TODAS las jornadas futuras
                    $stmt = $conn->prepare("
                        SELECT j.id, j.ciudad_nombre, j.fecha_jornada, r.nombre_ruta
                        FROM jornadas j
                        JOIN rutas r ON j.ruta_id = r.id
                        WHERE r.activa = 1
                        AND j.fecha_jornada >= CURDATE()
                        ORDER BY j.fecha_jornada ASC, r.nombre_ruta ASC
                    ");
                    $stmt->execute();
                } else {
                    // Si es operador normal, solo mostrar jornadas asignadas
                    $stmt = $conn->prepare("
                        SELECT j.id, j.ciudad_nombre, j.fecha_jornada, r.nombre_ruta
                        FROM jornadas j
                        JOIN jornada_operadores jo ON j.id = jo.jornada_id
                        JOIN rutas r ON j.ruta_id = r.id
                        WHERE r.activa = 1
                        AND jo.operador_id = ?
                        AND j.fecha_jornada >= CURDATE()
                        ORDER BY j.fecha_jornada ASC, r.nombre_ruta ASC
                    ");
                    $stmt->execute([$usuario_id]);
                }
                
                $jornadas = $stmt->fetchAll(PDO::FETCH_ASSOC);
                $response = ['success' => true, 'data' => $jornadas];

            } catch (PDOException $e) {
                // Te recomiendo mantener el error detallado mientras desarrollas
                $response['message'] = 'Error de base de datos: ' . $e->getMessage();
                error_log("Error en get_jornadas_asignadas_operador: " . $e->getMessage());
            }
            break;

        case 'verificar_acciones_pendientes':
            try {
                $stmt = $conn->prepare("
                    SELECT id, nombre, telefono, ultima_accion, fecha_proxima_accion 
                    FROM prospectos 
                    WHERE operador_id = ? 
                    AND fecha_proxima_accion IS NOT NULL 
                    AND fecha_proxima_accion <= NOW()
                    AND estado NOT IN ('asistido', 'no interesado', 'cancelo')
                ");
                $stmt->execute([$usuario_id]);
                $pendientes = $stmt->fetchAll(PDO::FETCH_ASSOC);
                $response = ['success' => true, 'data' => $pendientes];
            } catch (PDOException $e) {
                $response['message'] = 'Error al verificar alertas.';
            }
            break;

        case 'obtener_prospecto_individual':
            try {
                $id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
                if ($id) {
                    $stmt = $conn->prepare("SELECT * FROM prospectos WHERE id = ? AND operador_id = ?");
                    $stmt->execute([$id, $usuario_id]);
                    $prospecto = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if ($prospecto) {
                        $response = ['success' => true, 'data' => $prospecto];
                    } else {
                        $response['message'] = 'Prospecto no encontrado o sin permisos.';
                    }
                } else {
                    $response['message'] = 'ID de prospecto inválido.';
                }
            } catch (PDOException $e) {
                error_log("Error en obtener_prospecto_individual: " . $e->getMessage());
                $response['message'] = 'Error al cargar el prospecto.';
            }
            break;

        case 'agendar_prospecto_a_jornada':
            $prospecto_id = filter_input(INPUT_POST, 'prospecto_id', FILTER_VALIDATE_INT);
            $jornada_id = filter_input(INPUT_POST, 'jornada_id', FILTER_VALIDATE_INT);
            $cedula = trim($_POST['cedula'] ?? '');
            $edad = filter_input(INPUT_POST, 'edad', FILTER_VALIDATE_INT);
            $sexo = in_array($_POST['sexo'] ?? '', ['M', 'F']) ? $_POST['sexo'] : null;
            $hora_asistencia = $_POST['hora_asistencia'] ?? null;
            $notas_agendamiento = trim($_POST['notas'] ?? '');

            if (!$prospecto_id || !$jornada_id || empty($cedula) || !$edad || !$hora_asistencia || !$sexo) {
                $response['message'] = 'Faltan datos (Cédula, Edad, Sexo, Jornada, Hora).';
                break;
            }
            try {
                $conn->beginTransaction();
                $stmt_prospecto = $conn->prepare("SELECT * FROM prospectos WHERE id = ? AND operador_id = ?");
                $stmt_prospecto->execute([$prospecto_id, $usuario_id]);
                $prospecto = $stmt_prospecto->fetch(PDO::FETCH_ASSOC);

                if (!$prospecto) { $conn->rollBack(); $response['message'] = 'Prospecto no encontrado.'; break; }

                $stmt_find_cliente = $conn->prepare("SELECT id FROM clientes WHERE cedula = ? LIMIT 1");
                $stmt_find_cliente->execute([$cedula]);
                $cliente_id = $stmt_find_cliente->fetchColumn();

                if ($cliente_id) {
                    // Actualizar cliente existente incluyendo el sexo y ORIGEN
                    $stmt_update_cliente = $conn->prepare("UPDATE clientes SET nombre = ?, telefono = ?, ciudad = ?, edad = ?, sexo = ?, estado = ?, estado_global = ?, fecha_ultima_actualizacion = NOW() WHERE id = ?");
                    $stmt_update_cliente->execute([$prospecto['nombre'], $prospecto['telefono'], $prospecto['ciudad'], $edad, $sexo, 'agendado', 'Agendado en Jornada', $cliente_id]);
                } else {
                    // Crear nuevo cliente incluyendo el sexo y ORIGEN
                    $stmt_crear_cliente = $conn->prepare("INSERT INTO clientes (nombre, cedula, telefono, ciudad, edad, sexo, estado, estado_global, origen, origen_otro, creado_por_id, fecha_primer_contacto) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())");
                    $stmt_crear_cliente->execute([$prospecto['nombre'], $cedula, $prospecto['telefono'], $prospecto['ciudad'], $edad, $sexo, 'agendado', 'Agendado en Jornada', $prospecto['origen'] ?? null, $prospecto['origen_otro'] ?? null, $usuario_id]);
                    $cliente_id = $conn->lastInsertId();
                }

                $stmt_check_agenda = $conn->prepare("SELECT id FROM jornada_clientes WHERE jornada_id = ? AND cliente_id = ?");
                $stmt_check_agenda->execute([$jornada_id, $cliente_id]);
                if ($stmt_check_agenda->fetch()) { $conn->rollBack(); $response['message'] = 'Este cliente ya está agendado en la jornada.'; break; }

                $stmt_agendar = $conn->prepare("INSERT INTO jornada_clientes (jornada_id, cliente_id, agendado_por_operador_id, hora_asistencia, edad, sexo, estado_cita) VALUES (?, ?, ?, ?, ?, ?, 'Agendado')");
                $stmt_agendar->execute([$jornada_id, $cliente_id, $usuario_id, $hora_asistencia, $edad, $sexo]);

                $notas_actualizadas = $prospecto['notas'] . "\n---\n[".date('Y-m-d H:i:s')." - Agendamiento]\n" . $notas_agendamiento;
                $stmt_update_prospecto = $conn->prepare("UPDATE prospectos SET estado = 'agendado', ultima_accion = 'Agendado en Jornada', cedula = ?, notas = ?, fecha_ultima_llamada = NOW() WHERE id = ?");
                $stmt_update_prospecto->execute([$cedula, $notas_actualizadas, $prospecto_id]);
                
                $conn->commit();
                registrar_accion('agendamiento_prospecto', "El operador agendó al cliente ID {$cliente_id} (prospecto ID {$prospecto_id}) a la jornada ID {$jornada_id}.");
                $response = ['success' => true, 'message' => '¡Prospecto agendado! El contacto fue guardado/actualizado como cliente.'];
            } catch (Exception $e) {
                $conn->rollBack();
                $response['message'] = ($e->getCode() == '23000') ? 'Este cliente ya parece estar agendado.' : 'Error en el proceso: ' . $e->getMessage();
                error_log("Error en agendar_prospecto_a_jornada: " . $e->getMessage());
            }
            break;

        case 'actualizar_prospecto_operador':
            $prospecto_id = filter_input(INPUT_POST, 'prospecto_id', FILTER_VALIDATE_INT);
            $estado_nuevo = $_POST['estado'] ?? '';
            $accion_nueva = $_POST['ultima_accion'] ?? '';
            $nota_adicional = trim($_POST['notas'] ?? '');
            $nuevo_nombre = trim($_POST['nuevo_nombre'] ?? ''); // NUEVO: Campo de nombre opcional

            if (!$prospecto_id) {
                $response['message'] = 'ID de prospecto inválido.';
                break;
            }

            try {
                $stmt_check = $conn->prepare("SELECT nombre, telefono, cedula, estado, ultima_accion, notas, origen, origen_otro FROM prospectos WHERE id = ? AND operador_id = ?");
                $stmt_check->execute([$prospecto_id, $usuario_id]);
                $prospecto_actual = $stmt_check->fetch(PDO::FETCH_ASSOC);

                if (!$prospecto_actual) {
                    $response['message'] = 'No tienes permiso para modificar este prospecto.';
                    break;
                }

                $estado_actual = $prospecto_actual['estado'];
                $cedula_prospecto = $prospecto_actual['cedula'];
                $telefono_prospecto = $prospecto_actual['telefono'];
                
                // NUEVO: Determinar el nombre a usar
                $nombre_a_usar = $prospecto_actual['nombre']; // Por defecto usar el nombre actual
                if (!empty($nuevo_nombre)) {
                    $nombre_a_usar = $nuevo_nombre;
                    // Agregar nota sobre el cambio de nombre
                    if (strtolower(trim($prospecto_actual['nombre'])) === 'sin nombre') {
                        $nota_adicional .= "\n[Sistema: Nombre actualizado de 'Sin Nombre' a '{$nuevo_nombre}']";
                    } else {
                        $nota_adicional .= "\n[Sistema: Nombre actualizado de '{$prospecto_actual['nombre']}' a '{$nuevo_nombre}']";
                    }
                }
                
                if (empty($estado_nuevo) || empty($accion_nueva)) {
                    $response['message'] = 'El estado y la acción son obligatorios para guardar.';
                    break;
                }
                
                $conn->beginTransaction();

                // ===== LÓGICA PARA MANEJAR CAMBIOS DE ESTADO "AGENDADO" =====
                if ($estado_actual === 'agendado' && $estado_nuevo !== 'agendado' && !empty($cedula_prospecto)) {
                    $stmt_find_cliente = $conn->prepare("SELECT id FROM clientes WHERE cedula = ? LIMIT 1");
                    $stmt_find_cliente->execute([$cedula_prospecto]);
                    $cliente_id = $stmt_find_cliente->fetchColumn();
                    
                    if ($cliente_id) {
                        $stmt_delete = $conn->prepare(
                            "DELETE jc FROM jornada_clientes jc
                             JOIN jornadas j ON jc.jornada_id = j.id
                             WHERE jc.cliente_id = ? AND j.fecha_jornada >= CURDATE()"
                        );
                        $stmt_delete->execute([$cliente_id]);
                        $nota_adicional .= "\n[Sistema: Eliminado de la jornada por cambio de estado]";
                    }
                }

                // ===== ACTUALIZAR NOTAS DEL PROSPECTO =====
                $notas_para_guardar = $prospecto_actual['notas'];
                if (!empty($nota_adicional)) {
                    $notas_para_guardar .= "\n---\n[".date('Y-m-d H:i:s')."]\n".$nota_adicional;
                }

                // ===== CALCULAR FECHA PRÓXIMA ACCIÓN =====
                $fecha_proxima_accion = null;
                if (strpos($accion_nueva, 'Llamar en') !== false) {
                    if($accion_nueva === 'Llamar en 15 Min') $fecha_proxima_accion = date('Y-m-d H:i:s', strtotime('+15 minutes'));
                    if($accion_nueva === 'Llamar en 30 Min') $fecha_proxima_accion = date('Y-m-d H:i:s', strtotime('+30 minutes'));
                    if($accion_nueva === 'Llamar en 60 Min') $fecha_proxima_accion = date('Y-m-d H:i:s', strtotime('+60 minutes'));
                    if($accion_nueva === 'Llamar en 3 horas') $fecha_proxima_accion = date('Y-m-d H:i:s', strtotime('+3 hours'));
                    if($accion_nueva === 'Llamar en 24 horas') $fecha_proxima_accion = date('Y-m-d H:i:s', strtotime('+24 hours'));
                }

                // ===== ACTUALIZAR PROSPECTO (INCLUYENDO NOMBRE SI SE PROPORCIONÓ) =====
                $stmt_update = $conn->prepare("UPDATE prospectos SET nombre = ?, estado = ?, ultima_accion = ?, fecha_proxima_accion = ?, notas = ?, fecha_ultima_llamada = NOW() WHERE id = ?");
                $stmt_update->execute([$nombre_a_usar, $estado_nuevo, $accion_nueva, $fecha_proxima_accion, $notas_para_guardar, $prospecto_id]);

                // ===== SINCRONIZAR CON TABLA CLIENTES =====
                // Buscar el cliente por teléfono (y cédula si existe)
                $cliente_id = null;
                
                // Primero intentar buscar por cédula si existe
                if (!empty($cedula_prospecto)) {
                    $stmt_find_cliente = $conn->prepare("SELECT id FROM clientes WHERE cedula = ? LIMIT 1");
                    $stmt_find_cliente->execute([$cedula_prospecto]);
                    $cliente_id = $stmt_find_cliente->fetchColumn();
                }
                
                // Si no se encontró por cédula, buscar por teléfono
                if (!$cliente_id && !empty($telefono_prospecto)) {
                    $stmt_find_cliente = $conn->prepare("SELECT id FROM clientes WHERE telefono = ? LIMIT 1");
                    $stmt_find_cliente->execute([$telefono_prospecto]);
                    $cliente_id = $stmt_find_cliente->fetchColumn();
                }

                // El estado_global ahora guardará la acción del prospecto (ultima_accion)
                $estado_global_cliente = $accion_nueva;

                if ($cliente_id) {
                    // Actualizar cliente existente (usando el nombre actualizado)
                    $stmt_update_cliente = $conn->prepare("
                        UPDATE clientes 
                        SET nombre = ?, 
                            estado = ?, 
                            estado_global = ?, 
                            fecha_ultima_actualizacion = NOW() 
                        WHERE id = ?
                    ");
                    $stmt_update_cliente->execute([
                        $nombre_a_usar, 
                        $estado_nuevo, 
                        $estado_global_cliente, 
                        $cliente_id
                    ]);
                } else if (!empty($telefono_prospecto)) {
                    // Crear nuevo cliente si no existe (usando el nombre actualizado)
                    $stmt_create_cliente = $conn->prepare("
                        INSERT INTO clientes 
                        (nombre, telefono, cedula, estado, estado_global, origen, origen_otro, creado_por_id, fecha_primer_contacto) 
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())
                    ");
                    $stmt_create_cliente->execute([
                        $nombre_a_usar, 
                        $telefono_prospecto, 
                        $cedula_prospecto, 
                        $estado_nuevo, 
                        $estado_global_cliente, 
                        $prospecto_actual['origen'] ?? null,
                        $prospecto_actual['origen_otro'] ?? null,
                        $usuario_id
                    ]);
                    $cliente_id = $conn->lastInsertId();
                }

                // Registrar acción en log (usando el nombre actualizado)
                $detalle_log = "Actualizó prospecto '{$nombre_a_usar}' (ID: {$prospecto_id}). Estado: {$estado_nuevo}, Acción: {$accion_nueva}.";
                if (!empty($nuevo_nombre)) {
                    $detalle_log .= " Nombre actualizado de '{$prospecto_actual['nombre']}' a '{$nuevo_nombre}'.";
                }
                if ($cliente_id) {
                    $detalle_log .= " Sincronizado con cliente ID: {$cliente_id}.";
                }
                registrar_accion('actualiza_prospecto', $detalle_log);

                $conn->commit();
                
                // Mensaje de éxito personalizado
                $mensaje_exito = 'Prospecto actualizado y sincronizado correctamente.';
                if (!empty($nuevo_nombre)) {
                    $mensaje_exito = "Prospecto actualizado correctamente. Nombre cambiado a '{$nuevo_nombre}'.";
                }
                
                $response = ['success' => true, 'message' => $mensaje_exito];

            } catch(Exception $e) {
                $conn->rollBack();
                $response['message'] = 'Error al guardar los datos.';
                error_log("Error en actualizar_prospecto_operador: " . $e->getMessage());
            }
            break;
        
        case 'importar_mis_prospectos':
            if (isset($_FILES['archivo_csv']) && $_FILES['archivo_csv']['error'] == 0) {
                $archivo = $_FILES['archivo_csv']['tmp_name'];
                $handle = fopen($archivo, "r");
                fgetcsv($handle); 
                $importados = 0; $fallidos = 0;
                
                try {
                    $conn->beginTransaction();
                    $stmt_prospecto = $conn->prepare("INSERT INTO prospectos (nombre, telefono, ciudad, campana, origen, origen_otro, operador_id, admin_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
                    $stmt_check_cliente = $conn->prepare("SELECT id FROM clientes WHERE telefono = ? LIMIT 1");
                    $stmt_update_cliente = $conn->prepare("UPDATE clientes SET nombre = ?, ciudad = ?, estado = ?, estado_global = ?, fecha_ultima_actualizacion = NOW() WHERE id = ?");
                    $stmt_create_cliente = $conn->prepare("INSERT INTO clientes (nombre, telefono, ciudad, estado, estado_global, origen, origen_otro, creado_por_id, fecha_primer_contacto) VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())");
                    
                    while (($data = fgetcsv($handle, 1000, ",")) !== FALSE) {
                        if (count($data) >= 4) {
                            try { 
                                $nombre = !empty(trim($data[0])) ? trim($data[0]) : 'Sin Nombre';
                                $telefono = trim($data[1]);
                                $ciudad = trim($data[2]);
                                $campana = trim($data[3]);
                                
                                // Insertar prospecto
                                $stmt_prospecto->execute([$nombre, $telefono, $ciudad, $campana, null, null, $usuario_id, $usuario_id]);
                                
                                // Sincronizar con tabla clientes
                                $stmt_check_cliente->execute([$telefono]);
                                $cliente_existente = $stmt_check_cliente->fetchColumn();
                                
                                if ($cliente_existente) {
                                    $stmt_update_cliente->execute([$nombre, $ciudad, 'nuevo', 'Contacto importado', $cliente_existente]);
                                } else {
                                    $stmt_create_cliente->execute([$nombre, $telefono, $ciudad, 'nuevo', 'Contacto importado', null, null, $usuario_id]);
                                }
                                
                                $importados++; 
                            } catch (PDOException $e) { $fallidos++; }
                        } else { $fallidos++; }
                    }
                    
                    $conn->commit();
                    fclose($handle);
                    
                    if ($importados > 0) {
                        registrar_accion('importar_mis_prospectos', "Importó {$importados} nuevos prospectos y los sincronizó con la base de clientes.");
                    }
                    $response = ['success' => true, 'message' => "Importación completada. Exitosos: $importados, Fallidos: $fallidos. Todos sincronizados con base de clientes."];
                    
                } catch (Exception $e) {
                    $conn->rollBack();
                    fclose($handle);
                    $response['message'] = 'Error durante la importación: ' . $e->getMessage();
                }
            } else {
                $response['message'] = 'Error al subir el archivo.';
            }
            break;

        case 'agregar_prospecto_manual':
             try {
                $nombre = trim($_POST['nombre'] ?? '');
                $telefono = trim($_POST['telefono'] ?? '');
                $origen = trim($_POST['origen'] ?? '');
                $origen_otro = trim($_POST['origen_otro'] ?? '');

                if(empty($telefono)){
                    $response['message'] = 'El teléfono es obligatorio.';
                    break;
                }

                // Si no se envía nombre o está vacío, usar "Sin Nombre"
                if(empty($nombre)){
                    $nombre = 'Sin Nombre';
                }

                $conn->beginTransaction();
                
                // Insertar prospecto (AHORA INCLUYE ORIGEN Y ORIGEN_OTRO)
                $stmt = $conn->prepare("INSERT INTO prospectos (nombre, telefono, ciudad, campana, origen, origen_otro, operador_id, admin_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([$nombre, $telefono, $_POST['ciudad'] ?? null, $_POST['campana'] ?? null, $origen, $origen_otro, $usuario_id, $usuario_id]);
                
                // Sincronizar con tabla clientes (AHORA INCLUYE ORIGEN Y ORIGEN_OTRO)
                $stmt_check_cliente = $conn->prepare("SELECT id FROM clientes WHERE telefono = ? LIMIT 1");
                $stmt_check_cliente->execute([$telefono]);
                $cliente_existente = $stmt_check_cliente->fetchColumn();
                
                if ($cliente_existente) {
                    $stmt_update_cliente = $conn->prepare("UPDATE clientes SET nombre = ?, ciudad = ?, estado = ?, estado_global = ?, fecha_ultima_actualizacion = NOW() WHERE id = ?");
                    $stmt_update_cliente->execute([$nombre, $_POST['ciudad'] ?? null, 'nuevo', 'Agregado manualmente', $cliente_existente]);
                } else {
                    $stmt_create_cliente = $conn->prepare("INSERT INTO clientes (nombre, telefono, ciudad, estado, estado_global, origen, origen_otro, creado_por_id, fecha_primer_contacto) VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())");
                    $stmt_create_cliente->execute([$nombre, $telefono, $_POST['ciudad'] ?? null, 'nuevo', 'Agregado manualmente', $origen, $origen_otro, $usuario_id]);
                }
                
                $conn->commit();
                registrar_accion('agregar_prospecto_manual', "Agregó manualmente al prospecto '{$nombre}' y lo sincronizó con la base de clientes.");
                $response = ['success' => true, 'message' => 'Prospecto registrado exitosamente y sincronizado con base de clientes.'];

            } catch (PDOException $e) {
                $conn->rollBack();
                error_log("Error agregando prospecto manual: " . $e->getMessage());
                $response['message'] = 'Error al registrar el prospecto.';
            }
            break;
            
        default:
             $response['message'] = 'Acción de operador desconocida.';
             break;
    }
} else {
    $response['message'] = 'Acceso denegado o acción no reconocida.';
}

echo json_encode($response);
?>