<?php
define('IS_AJAX', true);
require_once '../administracion/includes/seguridad.php';

header('Content-Type: application/json');
$response = ['success' => false, 'message' => 'Acción no válida.'];
$usuario_id = $_SESSION['user_id'] ?? 0;
$rol_usuario = $_SESSION['rol_clave'] ?? ''; // Variable para verificar el rol

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        $response['message'] = 'Error de seguridad (CSRF).';
        echo json_encode($response);
        exit();
    }
}

$action = $_POST['action'] ?? $_GET['action'] ?? '';

switch ($action) {
    case 'buscar_clientes_reactivacion':
        if (!puede('ver_reactivacion_admin')) {
            $response['message'] = 'No tienes permiso para buscar clientes.';
            break;
        }

        try {
            $ciudad = trim($_POST['ciudad'] ?? '');
            $fecha_ultimo_contacto = trim($_POST['fecha_ultimo_contacto'] ?? '');
            $params = [];
            
            $sql = "SELECT id, nombre, cedula, telefono, ciudad, estado_global 
                    FROM clientes 
                    WHERE id NOT IN (SELECT cliente_id FROM reactivacion_clientes WHERE estado_contacto != 'Finalizado')";

            if (!empty($ciudad)) {
                $sql .= " AND ciudad LIKE ?";
                $params[] = "%" . $ciudad . "%";
            }
            if (!empty($fecha_ultimo_contacto)) {
                $sql .= " AND fecha_primer_contacto < ?";
                $params[] = $fecha_ultimo_contacto;
            }

            $sql .= " ORDER BY nombre ASC LIMIT 200";
            $stmt = $conn->prepare($sql);
            $stmt->execute($params);
            $clientes = $stmt->fetchAll(PDO::FETCH_ASSOC);

            $response = ['success' => true, 'data' => $clientes];

        } catch (PDOException $e) {
            $response['message'] = 'Error de base de datos al buscar clientes: ' . $e->getMessage();
            error_log($e->getMessage());
        }
        break;

    case 'crear_lote_reactivacion':
        if (!puede('crear_lote_reactivacion')) {
            $response['message'] = 'No tienes permiso para crear lotes.';
            break;
        }

        $nombre_lote = trim($_POST['nombre_lote'] ?? '');
        $supervisor_id = filter_input(INPUT_POST, 'supervisor_id', FILTER_VALIDATE_INT);
        $clientes_ids = json_decode($_POST['clientes_ids'] ?? '[]', true);

        if (empty($nombre_lote) || !$supervisor_id || empty($clientes_ids)) {
            $response['message'] = 'Faltan datos para crear el lote (Nombre, Supervisor o Clientes).';
            break;
        }

        try {
            $conn->beginTransaction();
            $stmt_lote = $conn->prepare("INSERT INTO reactivacion_lotes (nombre_lote, admin_id, supervisor_id) VALUES (?, ?, ?)");
            $stmt_lote->execute([$nombre_lote, $usuario_id, $supervisor_id]);
            $lote_id = $conn->lastInsertId();

            $stmt_clientes = $conn->prepare("INSERT INTO reactivacion_clientes (lote_id, cliente_id) VALUES (?, ?)");
            foreach ($clientes_ids as $cliente_id) {
                $stmt_clientes->execute([$lote_id, $cliente_id]);
            }

            $conn->commit();
            registrar_accion('crear_lote_reactivacion', "Creó el lote de reactivación '{$nombre_lote}' (ID: {$lote_id}) con " . count($clientes_ids) . " clientes.");
            $response = ['success' => true, 'message' => 'Lote de reactivación creado y asignado al supervisor correctamente.'];

        } catch (PDOException $e) {
            $conn->rollBack();
            $response['message'] = 'Error de base de datos al crear el lote: ' . $e->getMessage();
            error_log($e->getMessage());
        }
        break;

    case 'get_supervisores':
        try {
            $stmt = $conn->query("SELECT id, nombre FROM usuarios WHERE rol_id = 4 AND estado = 1 ORDER BY nombre ASC"); // Asumiendo que rol_id 4 es supervisor
            $supervisores = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $response = ['success' => true, 'data' => $supervisores];
        } catch (PDOException $e) {
            $response['message'] = 'No se pudieron cargar los supervisores.';
        }
        break;
        
    case 'get_lotes_supervisor':
        if (!puede('ver_reactivacion_supervisor')) {
            $response['message'] = 'No tienes permiso para ver lotes de reactivación.';
            break;
        }

        try {
            $sql_base = "
                SELECT 
                    rl.id, rl.nombre_lote, rl.fecha_creacion, rl.estado,
                    u_admin.nombre as admin_nombre,
                    u_super.nombre as supervisor_nombre,
                    (SELECT COUNT(*) FROM reactivacion_clientes rc WHERE rc.lote_id = rl.id) as total_clientes,
                    (SELECT COUNT(*) FROM reactivacion_clientes rc WHERE rc.lote_id = rl.id AND rc.operador_id IS NOT NULL) as clientes_asignados
                FROM reactivacion_lotes rl
                JOIN usuarios u_admin ON rl.admin_id = u_admin.id
                JOIN usuarios u_super ON rl.supervisor_id = u_super.id
            ";
            
            $params = [];
            // Si el usuario NO es superadmin, se filtra por su propio ID de supervisor
            if ($rol_usuario !== 'superadmin') {
                $sql_base .= " WHERE rl.supervisor_id = ?";
                $params[] = $usuario_id;
            }
            
            $sql_base .= " ORDER BY rl.fecha_creacion DESC";

            $stmt = $conn->prepare($sql_base);
            $stmt->execute($params);
            $lotes = $stmt->fetchAll(PDO::FETCH_ASSOC);

            $response = ['success' => true, 'data' => $lotes];

        } catch (PDOException $e) {
            $response['message'] = 'Error al cargar los lotes asignados: ' . $e->getMessage();
            error_log($e->getMessage());
        }
        break;

    case 'get_detalles_lote_supervisor':
        if (!puede('ver_reactivacion_supervisor')) { $response['message'] = 'Acceso denegado.'; break; }
        $lote_id = filter_input(INPUT_GET, 'lote_id', FILTER_VALIDATE_INT);
        if (!$lote_id) { $response['message'] = 'ID de lote no válido.'; break; }

        try {
            $sql_info = "SELECT nombre_lote FROM reactivacion_lotes WHERE id = ?";
            $params_info = [$lote_id];
            // Si NO es superadmin, se asegura que el lote le pertenezca
            if ($rol_usuario !== 'superadmin') {
                $sql_info .= " AND supervisor_id = ?";
                $params_info[] = $usuario_id;
            }
            $stmt_info = $conn->prepare($sql_info);
            $stmt_info->execute($params_info);
            $info_lote = $stmt_info->fetch(PDO::FETCH_ASSOC);

            if(!$info_lote) { $response['message'] = 'Lote no encontrado o sin permisos para verlo.'; break; }

            $stmt_clientes = $conn->prepare("
                SELECT 
                    rc.id as reactivacion_id, rc.cliente_id, rc.estado_contacto,
                    c.nombre, c.cedula, c.telefono,
                    u.nombre as operador_nombre
                FROM reactivacion_clientes rc
                JOIN clientes c ON rc.cliente_id = c.id
                LEFT JOIN usuarios u ON rc.operador_id = u.id
                WHERE rc.lote_id = ?
            ");
            $stmt_clientes->execute([$lote_id]);
            $clientes = $stmt_clientes->fetchAll(PDO::FETCH_ASSOC);

            // Calcular estadísticas
            $total = count($clientes);
            $asignados = count(array_filter($clientes, fn($c) => !empty($c['operador_nombre'])));
            $pendientes = $total - $asignados;
            $contactados = count(array_filter($clientes, fn($c) => $c['estado_contacto'] !== 'Pendiente'));
            $progreso = ($total > 0) ? round(($contactados / $total) * 100) : 0;

            $response['success'] = true;
            $response['data'] = [
                'info' => $info_lote,
                'clientes' => $clientes,
                'stats' => [ 'total' => $total, 'asignados' => $asignados, 'pendientes' => $pendientes, 'progreso' => $progreso ]
            ];

        } catch (PDOException $e) {
            $response['message'] = 'Error al cargar los detalles del lote: ' . $e->getMessage();
            error_log($e->getMessage());
        }
        break;

    case 'get_operadores_equipo':
        try {
            // Si es superadmin, ve a TODOS los operadores. Si no, solo ve a su equipo.
            if ($rol_usuario === 'superadmin') {
                $stmt = $conn->query("SELECT id, nombre FROM usuarios WHERE rol_id = 5 AND estado = 1 ORDER BY nombre ASC"); // rol_id 5 es operador
            } else {
                $stmt = $conn->prepare("SELECT id, nombre FROM usuarios WHERE supervisor_id = ? AND estado = 1 ORDER BY nombre ASC");
                $stmt->execute([$usuario_id]);
            }
            $operadores = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $response = ['success' => true, 'data' => $operadores];
        } catch (PDOException $e) {
            $response['message'] = 'No se pudo cargar el equipo de operadores.';
        }
        break;

    case 'asignar_clientes_operador':
        if (!puede('distribuir_lote_reactivacion')) { $response['message'] = 'Acceso denegado.'; break; }
        $lote_id = filter_input(INPUT_POST, 'lote_id', FILTER_VALIDATE_INT);
        $operador_id = filter_input(INPUT_POST, 'operador_id', FILTER_VALIDATE_INT);
        $reactivacion_ids = json_decode($_POST['reactivacion_ids'] ?? '[]', true);

        if (!$lote_id || !$operador_id || empty($reactivacion_ids)) { $response['message'] = 'Faltan datos.'; break; }

        try {
            $conn->beginTransaction();
            
            // La verificación de seguridad solo se aplica si el que asigna NO es superadmin
            if ($rol_usuario !== 'superadmin') {
                $stmt_check = $conn->prepare("SELECT supervisor_id FROM usuarios WHERE id = ?");
                $stmt_check->execute([$operador_id]);
                $supervisor_del_operador = $stmt_check->fetchColumn();

                if ($supervisor_del_operador != $usuario_id) {
                    $conn->rollBack();
                    $response['message'] = 'Error: Solo puedes asignar clientes a operadores de tu propio equipo.';
                    echo json_encode($response); exit();
                }
            }

            $placeholders = implode(',', array_fill(0, count($reactivacion_ids), '?'));
            $stmt = $conn->prepare("UPDATE reactivacion_clientes SET operador_id = ?, fecha_asignacion_operador = NOW() WHERE id IN ($placeholders) AND lote_id = ?");
            
            $params = array_merge([$operador_id], $reactivacion_ids, [$lote_id]);
            $stmt->execute($params);
            
            $conn->commit();
            registrar_accion('asignar_clientes_reactivacion', "Usuario ID {$usuario_id} asignó " . count($reactivacion_ids) . " clientes al operador ID {$operador_id} en el lote ID {$lote_id}.");
            $response = ['success' => true, 'message' => 'Clientes asignados correctamente.'];

        } catch (PDOException $e) {
            $conn->rollBack();
            $response['message'] = 'Error al asignar los clientes: ' . $e->getMessage();
        }
        break;

    default:
        $response['message'] = 'Acción de reactivación desconocida.';
        break;
}

echo json_encode($response);
?>