<?php
define('IS_AJAX', true);
require_once '../administracion/includes/seguridad.php';

header('Content-Type: application/json');
$response = ['success' => false, 'message' => 'Acción no válida.'];
$usuario_id = $_SESSION['user_id'] ?? 0;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        $response['message'] = 'Error de seguridad (CSRF).';
        echo json_encode($response);
        exit();
    }
}

$action = $_POST['action'] ?? $_GET['action'] ?? '';

switch ($action) {
    // --- ACCIONES DE ADMINISTRADOR ---
    case 'get_stock_ruta':
        $ruta_id = filter_input(INPUT_GET, 'ruta_id', FILTER_VALIDATE_INT);
        if (!$ruta_id) { $response['message'] = 'ID de ruta no válido.'; break; }
        try {
            $stmt = $conn->prepare("
                SELECT rs.producto_id, rs.cantidad, p.nombre, p.sku
                FROM ruta_stock rs
                JOIN almacen_productos p ON rs.producto_id = p.id
                WHERE rs.ruta_id = ? AND rs.cantidad > 0
            ");
            $stmt->execute([$ruta_id]);
            $stock = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $response = ['success' => true, 'data' => $stock];
        } catch (Exception $e) {
            $response['message'] = 'Error al obtener el stock de la ruta.';
        }
        break;
    
    case 'cargar_rutas_admin_separadas':
        if (!puede('gestionar_rutas_admin')) { $response['message'] = 'Acceso denegado.'; break; }
        try {
            // Consulta base que se reutilizará
            $query = "
                SELECT 
                    r.id, r.nombre_ruta, r.activa, r.archivada, r.fecha_creacion, r.fecha_ultima_modificacion, 
                    coord.nombre as coordinador_nombre,
                    doc.nombre as doctor_nombre,
                    chof.nombre as chofer_nombre,
                    (SELECT COUNT(DISTINCT j.id) FROM jornadas j WHERE j.ruta_id = r.id) as total_jornadas
                FROM rutas r 
                LEFT JOIN usuarios coord ON r.coordinador_id = coord.id
                LEFT JOIN usuarios doc ON r.doctor_id = doc.id
                LEFT JOIN usuarios chof ON r.chofer_id = chof.id
            ";
            
            // Obtener rutas activas (no archivadas)
            $stmt_activas = $conn->prepare($query . " WHERE r.archivada = 0 ORDER BY r.activa DESC, r.fecha_creacion DESC");
            $stmt_activas->execute();
            $activas_planificadas = $stmt_activas->fetchAll(PDO::FETCH_ASSOC);

            // Obtener rutas pasadas (archivadas)
            $stmt_pasadas = $conn->prepare($query . " WHERE r.archivada = 1 ORDER BY r.fecha_creacion DESC");
            $stmt_pasadas->execute();
            $pasadas = $stmt_pasadas->fetchAll(PDO::FETCH_ASSOC);

            $response = ['success' => true, 'data' => [
                'activas_planificadas' => $activas_planificadas,
                'pasadas' => $pasadas
            ]];
        } catch(PDOException $e) { 
            $response['message'] = 'Error al cargar rutas: ' . $e->getMessage(); 
            error_log($e->getMessage()); 
        }
        break;
    
    case 'archivar_ruta':
        if (!puede('gestionar_rutas_admin')) { $response['message'] = 'Acceso denegado.'; break; }
        
        $ruta_id = filter_input(INPUT_POST, 'ruta_id', FILTER_VALIDATE_INT);
        if (!$ruta_id) { 
            $response['message'] = 'ID de ruta no válido.'; 
            break; 
        }
        
        try {
            $stmt = $conn->prepare("UPDATE rutas SET archivada = 1, activa = 0 WHERE id = ?");
            $stmt->execute([$ruta_id]);
            registrar_accion('archivar_ruta', "Archivó la ruta ID: {$ruta_id}.");
            $response = ['success' => true, 'message' => 'Ruta archivada correctamente. Ya no aparecerá en la lista principal.'];
        } catch (Exception $e) { 
            $response['message'] = 'Error al archivar la ruta.'; 
            error_log($e->getMessage());
        }
        break;
    
    case 'cambiar_estado_ruta':
        if (!puede('gestionar_rutas_admin')) { $response['message'] = 'Acceso denegado.'; break; }
        $ruta_id = filter_input(INPUT_POST, 'ruta_id', FILTER_VALIDATE_INT);
        if (!$ruta_id) { $response['message'] = 'ID de ruta no válido.'; break; }
        try {
            $stmt = $conn->prepare("UPDATE rutas SET activa = 1 - activa WHERE id = ?");
            $stmt->execute([$ruta_id]);
            registrar_accion('cambiar_estado_ruta', "Cambió el estado de activación de la ruta ID: {$ruta_id}.");
            $response = ['success' => true, 'message' => 'Estado de la ruta actualizado.'];
        } catch (Exception $e) { $response['message'] = 'Error al cambiar el estado de la ruta.'; }
        break;

    case 'crear_ruta_completa_simple':
        if (!puede('gestionar_rutas_admin')) { $response['message'] = 'Acceso denegado.'; break; }
        $nombre_ruta = trim($_POST['nombre_ruta'] ?? '');
        $coordinador_id = !empty($_POST['coordinador_id']) ? filter_var($_POST['coordinador_id'], FILTER_VALIDATE_INT) : null;
        $doctor_id = !empty($_POST['doctor_id']) ? filter_var($_POST['doctor_id'], FILTER_VALIDATE_INT) : null;
        $chofer_id = !empty($_POST['chofer_id']) ? filter_var($_POST['chofer_id'], FILTER_VALIDATE_INT) : null;
        $jornadas = $_POST['jornadas'] ?? [];

        if (empty($nombre_ruta) || empty($jornadas)) { $response['message'] = 'El nombre de la ruta y al menos una jornada son obligatorios.'; break; }
        try {
            $conn->beginTransaction();
            $stmt_ruta = $conn->prepare("INSERT INTO rutas (nombre_ruta, coordinador_id, doctor_id, chofer_id, fecha_creacion) VALUES (?, ?, ?, ?, NOW())");
            $stmt_ruta->execute([$nombre_ruta, $coordinador_id, $doctor_id, $chofer_id]);
            $ruta_id = $conn->lastInsertId();

            $stmt_jornada = $conn->prepare("INSERT INTO jornadas (ruta_id, ciudad_nombre, fecha_jornada) VALUES (?, ?, ?)");
            foreach($jornadas as $j) {
                if (!empty($j['ciudad']) && !empty($j['fecha'])) {
                    $stmt_jornada->execute([$ruta_id, trim($j['ciudad']), $j['fecha']]);
                }
            }
            $conn->commit();
            registrar_accion('crear_ruta', "Creó la ruta '{$nombre_ruta}' (ID: {$ruta_id}).");
            $response = ['success' => true, 'message' => 'Ruta creada. Ahora puedes asignar operadores desde el módulo de asignación.'];
        } catch (Exception $e) { $conn->rollBack(); $response['message'] = 'Error al guardar la ruta: ' . $e->getMessage(); }
        break;
        
        case 'get_rutas_para_asignacion':
        if (!puede('gestionar_rutas_admin')) { $response['message'] = 'Acceso denegado.'; break; }
        try {
            $stmt = $conn->query("SELECT id, nombre_ruta FROM rutas ORDER BY activa DESC, fecha_creacion DESC");
            $rutas = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $response = ['success' => true, 'data' => $rutas];
        } catch(PDOException $e) { $response['message'] = 'Error al cargar las rutas.'; }
        break;
        
        case 'guardar_asignacion_jornada': // <-- NUEVA ACCIÓN MÁS EFICIENTE
        if (!puede('gestionar_rutas_admin')) { $response['message'] = 'Acceso denegado.'; break; }
        $jornada_id = filter_input(INPUT_POST, 'jornada_id', FILTER_VALIDATE_INT);
        $operadores_ids = $_POST['operadores_ids'] ?? [];

        if (!$jornada_id) { $response['message'] = 'ID de jornada no válido.'; break; }
        
        try {
            $conn->beginTransaction();
            
            // Borrar solo las asignaciones de ESTA jornada
            $stmt_delete = $conn->prepare("DELETE FROM jornada_operadores WHERE jornada_id = ?");
            $stmt_delete->execute([$jornada_id]);

            // Insertar las nuevas asignaciones
            if (!empty($operadores_ids)) {
                $stmt_insert = $conn->prepare("INSERT INTO jornada_operadores (jornada_id, operador_id) VALUES (?, ?)");
                foreach ($operadores_ids as $operador_id) {
                    $stmt_insert->execute([$jornada_id, $operador_id]);
                }
            }
            
            $conn->commit();
            registrar_accion('asignar_operadores_jornada', "Actualizó asignaciones para la jornada ID: {$jornada_id}.");
            $response = ['success' => true, 'message' => 'Equipo actualizado.'];

        } catch (Exception $e) {
            $conn->rollBack();
            $response['message'] = 'Error al guardar las asignaciones: ' . $e->getMessage();
        }
        break;

    case 'get_jornadas_para_asignacion':
        if (!puede('gestionar_rutas_admin')) { $response['message'] = 'Acceso denegado.'; break; }
        $ruta_id = filter_input(INPUT_GET, 'ruta_id', FILTER_VALIDATE_INT);
        if (!$ruta_id) { $response['message'] = 'ID de ruta no válido.'; break; }
        try {
            $stmt_jornadas = $conn->prepare("SELECT id, ciudad_nombre, fecha_jornada FROM jornadas WHERE ruta_id = ? ORDER BY fecha_jornada ASC");
            $stmt_jornadas->execute([$ruta_id]);
            $jornadas = $stmt_jornadas->fetchAll(PDO::FETCH_ASSOC);

            $stmt_asignados = $conn->prepare("SELECT jornada_id, operador_id FROM jornada_operadores WHERE jornada_id IN (SELECT id FROM jornadas WHERE ruta_id = ?)");
            $stmt_asignados->execute([$ruta_id]);
            $asignados_raw = $stmt_asignados->fetchAll(PDO::FETCH_ASSOC);

            $asignados_map = [];
            foreach($asignados_raw as $asig) {
                $asignados_map[$asig['jornada_id']][] = $asig['operador_id'];
            }

            foreach($jornadas as &$jornada) {
                $jornada['operadores_asignados'] = $asignados_map[$jornada['id']] ?? [];
            }
            
            $response = ['success' => true, 'data' => $jornadas];
        } catch(PDOException $e) { $response['message'] = 'Error al cargar las jornadas.'; error_log($e->getMessage()); }
        break;

    case 'guardar_asignaciones_operadores':
        if (!puede('gestionar_rutas_admin')) { $response['message'] = 'Acceso denegado.'; break; }
        $ruta_id = filter_input(INPUT_POST, 'ruta_id', FILTER_VALIDATE_INT);
        $asignaciones = json_decode($_POST['asignaciones'] ?? '[]', true);
        if (!$ruta_id || empty($asignaciones)) { $response['message'] = 'Datos insuficientes.'; break; }

        try {
            $conn->beginTransaction();
            
            // Borramos solo las asignaciones de las jornadas de esta ruta
            $jornadas_ids = array_keys($asignaciones);
            $placeholders = implode(',', array_fill(0, count($jornadas_ids), '?'));
            $stmt_delete = $conn->prepare("DELETE FROM jornada_operadores WHERE jornada_id IN ($placeholders)");
            $stmt_delete->execute($jornadas_ids);

            // Insertamos las nuevas asignaciones
            $stmt_insert = $conn->prepare("INSERT INTO jornada_operadores (jornada_id, operador_id) VALUES (?, ?)");
            foreach ($asignaciones as $jornada_id => $operadores_ids) {
                if (is_array($operadores_ids)) {
                    foreach ($operadores_ids as $operador_id) {
                        $stmt_insert->execute([$jornada_id, $operador_id]);
                    }
                }
            }
            
            $conn->commit();
            registrar_accion('asignar_operadores', "Actualizó las asignaciones de operadores para la ruta ID: {$ruta_id}.");
            $response = ['success' => true, 'message' => 'Asignaciones guardadas con éxito.'];
        } catch (Exception $e) {
            $conn->rollBack();
            $response['message'] = 'Error al guardar las asignaciones: ' . $e->getMessage();
            error_log("Error en guardar_asignaciones_operadores: " . $e->getMessage());
        }
        break;


    case 'get_ruta_for_edit':
        if (!puede('gestionar_rutas_admin')) { $response['message'] = 'Acceso denegado.'; break; }
        $ruta_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
        if (!$ruta_id) { $response['message'] = 'ID de ruta no válido.'; break; }
        try {
            $stmt_ruta = $conn->prepare("SELECT r.*, u.nombre as coordinador_nombre FROM rutas r LEFT JOIN usuarios u ON r.coordinador_id = u.id WHERE r.id = ?");
            $stmt_ruta->execute([$ruta_id]);
            $ruta = $stmt_ruta->fetch(PDO::FETCH_ASSOC);

            if (!$ruta) {
                 $response['message'] = 'Ruta no encontrada.'; break;
            }

            $stmt_jornadas = $conn->prepare("SELECT * FROM jornadas WHERE ruta_id = ? ORDER BY fecha_jornada ASC");
            $stmt_jornadas->execute([$ruta_id]);
            $jornadas = $stmt_jornadas->fetchAll(PDO::FETCH_ASSOC);

            $stmt_pacientes = $conn->prepare("
                SELECT 
                    jc.jornada_id, 
                    c.id, c.nombre, c.cedula, c.telefono,
                    jc.id as jornada_cliente_id,
                    jc.estado_cita, jc.total_facturado,
                    jc.metodo_pago, jc.comprobante_pago, jc.notas_coordinador
                FROM jornada_clientes jc 
                JOIN clientes c ON jc.cliente_id = c.id 
                WHERE jc.jornada_id IN (SELECT id FROM jornadas WHERE ruta_id = ?)
            ");
            $stmt_pacientes->execute([$ruta_id]);
            $pacientes_por_jornada = $stmt_pacientes->fetchAll(PDO::FETCH_GROUP | PDO::FETCH_ASSOC);
            
            $hoy = new DateTime(); $hoy->setTime(0, 0, 0);

            $stmt_examenes = $conn->prepare("SELECT nombre_examen, precio FROM jornada_examenes WHERE jornada_cliente_id = ?");
            $stmt_medicamentos = $conn->prepare("SELECT nombre_medicamento, precio FROM jornada_medicamentos WHERE jornada_cliente_id = ?");

            foreach ($jornadas as &$jornada) {
                $pacientes_de_esta_jornada = $pacientes_por_jornada[$jornada['id']] ?? [];
                
                foreach($pacientes_de_esta_jornada as &$paciente) {
                    $stmt_examenes->execute([$paciente['jornada_cliente_id']]);
                    $paciente['examenes'] = $stmt_examenes->fetchAll(PDO::FETCH_ASSOC);
                    $stmt_medicamentos->execute([$paciente['jornada_cliente_id']]);
                    $paciente['medicamentos'] = $stmt_medicamentos->fetchAll(PDO::FETCH_ASSOC);
                }
                $jornada['pacientes'] = $pacientes_de_esta_jornada;

                $fecha_jornada = new DateTime($jornada['fecha_jornada']);
                $total_pacientes = count($jornada['pacientes']);
                $atendidos = 0;
                if($total_pacientes > 0) {
                    $atendidos = count(array_filter($jornada['pacientes'], function($p){
                        return $p['estado_cita'] === 'Atendido';
                    }));
                }

                if ($total_pacientes > 0 && $atendidos === $total_pacientes) {
                    $jornada['estado_jornada'] = 'Completada';
                } elseif ($fecha_jornada == $hoy) {
                    $jornada['estado_jornada'] = 'En Curso';
                } else {
                    $jornada['estado_jornada'] = 'Planificada';
                }
            }
            $ruta['jornadas'] = $jornadas;

            $response = ['success' => true, 'data' => $ruta];

        } catch (Exception $e) { $response['message'] = 'Error al obtener datos de la ruta: '.$e->getMessage(); error_log($e->getMessage()); }
        break;
        
    case 'editar_ruta_completa':
        if (!puede('gestionar_rutas_admin')) { $response['message'] = 'Acceso denegado.'; break; }
        
        $ruta_id = filter_input(INPUT_POST, 'ruta_id', FILTER_VALIDATE_INT);
        $nombre_ruta_nuevo = trim($_POST['nombre_ruta'] ?? '');
        $coordinador_id_nuevo = !empty($_POST['coordinador_id']) ? filter_var($_POST['coordinador_id'], FILTER_VALIDATE_INT) : null;
        $jornadas_nuevas = $_POST['jornadas'] ?? [];

        if (!$ruta_id || empty($nombre_ruta_nuevo)) {
            $response['message'] = 'Faltan datos esenciales (ID o nombre de ruta).';
            break;
        }

        try {
            $conn->beginTransaction();
            
            $stmt_old_ruta = $conn->prepare("SELECT r.nombre_ruta, r.coordinador_id, u.nombre as coordinador_nombre FROM rutas r LEFT JOIN usuarios u ON r.coordinador_id = u.id WHERE r.id = ?");
            $stmt_old_ruta->execute([$ruta_id]);
            $old_ruta = $stmt_old_ruta->fetch(PDO::FETCH_ASSOC);

            $stmt_old_jornadas = $conn->prepare("SELECT id, ciudad_nombre, fecha_jornada FROM jornadas WHERE ruta_id = ?");
            $stmt_old_jornadas->execute([$ruta_id]);
            $old_jornadas_map = $stmt_old_jornadas->fetchAll(PDO::FETCH_GROUP | PDO::FETCH_UNIQUE);

            // CAMBIO IMPORTANTE: Obtener pacientes con su información completa
            $stmt_old_pacientes = $conn->prepare("
                SELECT 
                    jc.jornada_id, 
                    c.id, c.nombre,
                    jc.id as jornada_cliente_id,
                    jc.estado_cita, 
                    jc.total_facturado,
                    jc.metodo_pago, 
                    jc.comprobante_pago, 
                    jc.notas_coordinador
                FROM jornada_clientes jc 
                JOIN clientes c ON jc.cliente_id = c.id 
                WHERE jc.jornada_id IN (SELECT id FROM jornadas WHERE ruta_id = ?)
            ");
            $stmt_old_pacientes->execute([$ruta_id]);
            $old_pacientes_map = $stmt_old_pacientes->fetchAll(PDO::FETCH_GROUP);
            
            $detalles_log = [];

            if ($old_ruta['nombre_ruta'] !== $nombre_ruta_nuevo) {
                $detalles_log[] = "Nombre de ruta cambiado de '{$old_ruta['nombre_ruta']}' a '{$nombre_ruta_nuevo}'";
            }
            if ($old_ruta['coordinador_id'] != $coordinador_id_nuevo) {
                $nuevo_coordinador_nombre = 'nadie';
                if ($coordinador_id_nuevo) {
                    $stmt_coord = $conn->prepare("SELECT nombre FROM usuarios WHERE id = ?");
                    $stmt_coord->execute([$coordinador_id_nuevo]);
                    $nuevo_coordinador_nombre = $stmt_coord->fetchColumn();
                }
                $detalles_log[] = "Coordinador cambiado de '".($old_ruta['coordinador_nombre'] ?? 'nadie')."' a '{$nuevo_coordinador_nombre}'";
            }
            
            $conn->prepare("UPDATE rutas SET nombre_ruta = ?, coordinador_id = ? WHERE id = ?")->execute([$nombre_ruta_nuevo, $coordinador_id_nuevo, $ruta_id]);

            $ids_jornadas_procesadas = [];
            $stmt_jornada_update = $conn->prepare("UPDATE jornadas SET ciudad_nombre = ?, fecha_jornada = ? WHERE id = ?");
            $stmt_jornada_insert = $conn->prepare("INSERT INTO jornadas (ruta_id, ciudad_nombre, fecha_jornada) VALUES (?, ?, ?)");
            $stmt_paciente_insert = $conn->prepare("INSERT INTO jornada_clientes (jornada_id, cliente_id) VALUES (?, ?)");

            foreach($jornadas_nuevas as $j) {
                if (empty($j['ciudad']) || empty($j['fecha'])) continue;

                $fecha_db = DateTime::createFromFormat('d/m/Y', $j['fecha']);
                if (!$fecha_db) { $fecha_db = DateTime::createFromFormat('Y-m-d', $j['fecha']); }
                if (!$fecha_db) { throw new Exception("Formato de fecha inválido."); }
                $fecha_sql = $fecha_db->format('Y-m-d');

                $jornada_id_actual = $j['id'];
                $pacientes_nuevos_ids = array_map('intval', $j['pacientes'] ?? []);

                if (strpos($jornada_id_actual, 'new_') === 0) {
                    // Nueva jornada
                    $stmt_jornada_insert->execute([$ruta_id, trim($j['ciudad']), $fecha_sql]);
                    $jornada_id_actual = $conn->lastInsertId();
                    $detalles_log[] = "Jornada AGREGADA: '{$j['ciudad']}' el {$j['fecha']}";
                    
                    // Para jornadas nuevas, agregar todos los pacientes sin información previa
                    foreach($pacientes_nuevos_ids as $paciente_id) {
                        $stmt_paciente_insert->execute([$jornada_id_actual, $paciente_id]);
                    }
                } else {
                    // Jornada existente
                    $jornada_id_actual = (int)$jornada_id_actual;
                    if (isset($old_jornadas_map[$jornada_id_actual])) {
                        $old_jornada = $old_jornadas_map[$jornada_id_actual];
                        if ($old_jornada['ciudad_nombre'] !== trim($j['ciudad'])) {
                            $detalles_log[] = "Jornada '{$old_jornada['ciudad_nombre']}' renombrada a '{$j['ciudad']}'";
                        }
                        $old_fecha_formato = (new DateTime($old_jornada['fecha_jornada']))->format('d/m/Y');
                        if ($old_fecha_formato !== $j['fecha']) {
                            $detalles_log[] = "Fecha de '{$j['ciudad']}' cambiada de {$old_fecha_formato} a {$j['fecha']}";
                        }
                    }
                    $stmt_jornada_update->execute([trim($j['ciudad']), $fecha_sql, $jornada_id_actual]);

                    // CAMBIO IMPORTANTE: Manejo inteligente de pacientes
                    $pacientes_antiguos = isset($old_pacientes_map[$jornada_id_actual]) ? $old_pacientes_map[$jornada_id_actual] : [];
                    $pacientes_antiguos_ids = array_column($pacientes_antiguos, 'id');
                    
                    // Crear un mapa de pacientes antiguos por ID para acceso rápido
                    $pacientes_antiguos_por_id = [];
                    foreach($pacientes_antiguos as $pac) {
                        $pacientes_antiguos_por_id[$pac['id']] = $pac;
                    }

                    $pacientes_agregados = array_diff($pacientes_nuevos_ids, $pacientes_antiguos_ids);
                    $pacientes_eliminados = array_diff($pacientes_antiguos_ids, $pacientes_nuevos_ids);

                    // Eliminar solo los pacientes que ya no están en la lista
                    if(!empty($pacientes_eliminados)) {
                        $nombres_eliminados = [];
                        foreach($pacientes_eliminados as $id_eliminado) {
                            if(isset($pacientes_antiguos_por_id[$id_eliminado])) {
                                $nombres_eliminados[] = $pacientes_antiguos_por_id[$id_eliminado]['nombre'];
                            }
                        }
                        $conn->prepare("DELETE FROM jornada_clientes WHERE jornada_id = ? AND cliente_id IN (".implode(',', $pacientes_eliminados).")")->execute([$jornada_id_actual]);
                        $detalles_log[] = "En '{$j['ciudad']}' se ELIMINARON pacientes: " . implode(', ', $nombres_eliminados);
                    }

                    // Agregar solo los pacientes nuevos (conservando la información de los existentes)
                    if(!empty($pacientes_agregados)) {
                        $nombres_agregados = $conn->query("SELECT nombre FROM clientes WHERE id IN (".implode(',', $pacientes_agregados).")")->fetchAll(PDO::FETCH_COLUMN);
                        foreach($pacientes_agregados as $paciente_id) {
                            $stmt_paciente_insert->execute([$jornada_id_actual, $paciente_id]);
                        }
                        $detalles_log[] = "En '{$j['ciudad']}' se AGREGARON pacientes: " . implode(', ', $nombres_agregados);
                    }

                    // Los pacientes que permanecen conservan toda su información (estado_cita, total_facturado, etc.)
                    $pacientes_conservados = array_intersect($pacientes_antiguos_ids, $pacientes_nuevos_ids);
                    if(!empty($pacientes_conservados)) {
                        $nombres_conservados = [];
                        foreach($pacientes_conservados as $id_conservado) {
                            if(isset($pacientes_antiguos_por_id[$id_conservado])) {
                                $pac = $pacientes_antiguos_por_id[$id_conservado];
                                if($pac['estado_cita'] === 'Atendido') {
                                    $nombres_conservados[] = $pac['nombre'];
                                }
                            }
                        }
                        if(!empty($nombres_conservados)) {
                            $detalles_log[] = "En '{$j['ciudad']}' se CONSERVARON pacientes atendidos: " . implode(', ', $nombres_conservados);
                        }
                    }
                }
                $ids_jornadas_procesadas[] = $jornada_id_actual;
            }
            
            // Eliminar jornadas que ya no están en la lista
            if (!empty($ids_jornadas_procesadas)) {
                $ids_jornadas_antiguas = array_keys($old_jornadas_map);
                $ids_jornadas_eliminadas = array_diff($ids_jornadas_antiguas, $ids_jornadas_procesadas);
                foreach($ids_jornadas_eliminadas as $id_eliminada) {
                    $detalles_log[] = "Jornada ELIMINADA: '{$old_jornadas_map[$id_eliminada]['ciudad_nombre']}'";
                }
                if (!empty($ids_jornadas_eliminadas)) {
                    $conn->prepare("DELETE FROM jornadas WHERE id IN (".implode(',', $ids_jornadas_eliminadas).")")->execute();
                }
            } else {
                if (!empty($old_jornadas_map)) {
                    $detalles_log[] = "Todas las jornadas fueron eliminadas.";
                    $conn->prepare("DELETE FROM jornadas WHERE ruta_id = ?")->execute([$ruta_id]);
                }
            }

            $conn->commit();
            
            if (!empty($detalles_log)) {
                registrar_accion('editar_ruta', "Editó la ruta '{$nombre_ruta_nuevo}' (ID: {$ruta_id}). Cambios: " . implode('; ', $detalles_log) . '.');
            }

            $response = ['success' => true, 'message' => 'Ruta actualizada correctamente manteniendo información de pacientes atendidos.'];

        } catch (Exception $e) {
            $conn->rollBack();
            $response['message'] = 'Error al actualizar la ruta: ' . $e->getMessage();
            error_log("Error en editar_ruta_completa: " . $e->getMessage());
        }
        break;

    case 'agregar_cliente_simple':
        if (!puede('gestionar_clientes')) { $response['message'] = 'Acceso denegado.'; break; }
        try {
            $nombre = trim($_POST['nombre'] ?? '');
            $cedula = trim($_POST['cedula'] ?? '');
            $telefono = trim($_POST['telefono'] ?? '');
            if (empty($nombre) || empty($cedula)) {
                $response['message'] = 'El nombre y la cédula son obligatorios.';
                echo json_encode($response); exit();
            }

            $stmt_check = $conn->prepare("SELECT id FROM clientes WHERE cedula = ?");
            $stmt_check->execute([$cedula]);
            if ($stmt_check->fetch()) {
                $response['message'] = 'Ya existe un cliente con esa cédula.';
                echo json_encode($response); exit();
            }

            $stmt = $conn->prepare("INSERT INTO clientes (nombre, cedula, telefono, creado_por_usuario_id) VALUES (?, ?, ?, ?)");
            $stmt->execute([$nombre, $cedula, $telefono, $usuario_id]);
            $new_id = $conn->lastInsertId();
            
            $response = [
                'success' => true, 
                'message' => 'Cliente registrado.',
                'data' => [
                    'id' => (int)$new_id,
                    'nombre' => $nombre,
                    'cedula' => $cedula,
                    'telefono' => $telefono
                ]
            ];
        } catch (Exception $e) {
            $response['message'] = 'Error al registrar el cliente.';
        }
        break;
        
        case 'get_todas_rutas_superadmin':
    // Solo superadmin puede usar esta acción
    if ($_SESSION['rol_clave'] !== 'superadmin') { 
        $response['message'] = 'Acceso denegado. Solo superadmin puede usar esta función.'; 
        break; 
    }
    
    try {
        // Obtener TODAS las rutas del sistema con información del coordinador
        $stmt = $conn->prepare("
            SELECT 
                r.id, 
                r.nombre_ruta, 
                r.coordinador_id,
                u.nombre as coordinador_nombre
            FROM rutas r 
            LEFT JOIN usuarios u ON r.coordinador_id = u.id 
            ORDER BY r.fecha_creacion DESC
        ");
        $stmt->execute();
        $rutas = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Obtener jornadas con información adicional para cada ruta
        $stmt_jornadas = $conn->prepare("
            SELECT 
                j.id, 
                j.ruta_id,
                j.ciudad_nombre, 
                j.fecha_jornada,
                COUNT(jc.id) as total_pacientes,
                COUNT(CASE WHEN jc.estado_cita = 'Atendido' THEN 1 END) as pacientes_atendidos
            FROM jornadas j 
            LEFT JOIN jornada_clientes jc ON j.id = jc.jornada_id
            WHERE j.ruta_id = ? 
            GROUP BY j.id, j.ruta_id, j.ciudad_nombre, j.fecha_jornada
            ORDER BY j.fecha_jornada ASC
        ");
        
        $hoy = new DateTime(); 
        $hoy->setTime(0,0,0);

        foreach($rutas as &$ruta) {
            $stmt_jornadas->execute([$ruta['id']]);
            $jornadas = $stmt_jornadas->fetchAll(PDO::FETCH_ASSOC);
            
            foreach($jornadas as &$jornada) {
                $fecha_jornada = new DateTime($jornada['fecha_jornada']);
                
                // Determinar estado de la jornada
                if ($fecha_jornada < $hoy) {
                    $jornada['estado_jornada'] = 'Completada';
                } elseif ($fecha_jornada == $hoy) {
                    $jornada['estado_jornada'] = 'En Curso';
                } else {
                    $jornada['estado_jornada'] = 'Planificada';
                }
                
                // Información adicional para supervisión
                $jornada['progreso'] = $jornada['total_pacientes'] > 0 ? 
                    round(($jornada['pacientes_atendidos'] / $jornada['total_pacientes']) * 100) : 0;
            }
            $ruta['jornadas'] = $jornadas;
        }

        $response = ['success' => true, 'data' => $rutas];
        
        // Registrar acceso del superadmin
        registrar_accion('supervision_rutas', "Superadmin accedió a la vista de supervisión de todas las rutas");
        
    } catch(PDOException $e) { 
        $response['message'] = 'Error al cargar rutas para supervisión.'; 
        error_log($e->getMessage()); 
    }
    break;
    

    case 'asignar_coordinador':
        if (!puede('gestionar_rutas_admin')) { $response['message'] = 'Acceso denegado.'; break; }
        try {
            $stmt = $conn->prepare("UPDATE rutas SET coordinador_id = ? WHERE id = ?");
            $stmt->execute([$_POST['coordinador_id'], $_POST['ruta_id']]);
            registrar_accion('asignar_coordinador', "Asignó un coordinador a la ruta ID: {$_POST['ruta_id']}.");
            $response = ['success' => true, 'message' => 'Coordinador asignado.'];
        } catch(PDOException $e) { $response['message'] = 'Error al asignar coordinador.'; error_log($e->getMessage()); }
        break;
        
    // --- NUEVAS ACCIONES PARA EL COORDINADOR ---
    case 'get_rutas_coordinador':
    if (!puede('ver_rutas_coordinador')) { $response['message'] = 'Acceso denegado.'; break; }
    try {
        // --- CORRECCIÓN --- Se añade "AND activa = 1" a la consulta
        $stmt = $conn->prepare("SELECT id, nombre_ruta FROM rutas WHERE coordinador_id = ? AND activa = 1 ORDER BY fecha_creacion DESC");
        $stmt->execute([$usuario_id]);
        $rutas = $stmt->fetchAll(PDO::FETCH_ASSOC);

        $stmt_jornadas = $conn->prepare("SELECT id, ciudad_nombre, fecha_jornada FROM jornadas WHERE ruta_id = ? ORDER BY fecha_jornada ASC");
        $hoy = new DateTime(); $hoy->setTime(0,0,0);

        foreach($rutas as &$ruta) {
            $stmt_jornadas->execute([$ruta['id']]);
            $jornadas = $stmt_jornadas->fetchAll(PDO::FETCH_ASSOC);
            foreach($jornadas as &$jornada) {
                $fecha_jornada = new DateTime($jornada['fecha_jornada']);
                // Se mantiene la lógica para determinar si la jornada está en curso o planificada
                $jornada['estado_jornada'] = ($fecha_jornada == $hoy) ? 'En Curso' : 'Planificada';
            }
            $ruta['jornadas'] = $jornadas;
        }
        $response = ['success' => true, 'data' => $rutas];
    } catch(PDOException $e) { 
        $response['message'] = 'Error al cargar tus rutas.'; 
        error_log($e->getMessage()); 
    }
    break;


    case 'get_jornada_coordinador':
    $jornada_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
    if (!$jornada_id) { 
        $response['message'] = 'ID de jornada no válido.'; 
        break; 
    }
    
    try {
        // Lógica de seguridad para verificar acceso del coordinador
        $stmt_check = $conn->prepare("SELECT j.id, j.fecha_jornada, r.coordinador_id FROM jornadas j JOIN rutas r ON j.ruta_id = r.id WHERE j.id = ?");
        $stmt_check->execute([$jornada_id]);
        $jornada_info = $stmt_check->fetch(PDO::FETCH_ASSOC);
        
        if (!$jornada_info) {
            $response['message'] = 'Jornada no encontrada.';
            echo json_encode($response); exit();
        }
        
        if ($jornada_info['coordinador_id'] != $usuario_id) {
            $response['message'] = 'No tienes acceso a esta jornada.';
            echo json_encode($response); exit();
        }
        
        $hoy = date('Y-m-d');
        if ($jornada_info['fecha_jornada'] != $hoy) {
            $response['message'] = 'Solo puedes acceder a jornadas en el día actual.';
            echo json_encode($response); exit();
        }

        // Consulta principal de pacientes (CORREGIDA)
        $stmt = $conn->prepare("
            SELECT 
                c.id as cliente_id, c.nombre, c.cedula, c.telefono, 
                jc.id as jornada_cliente_id, jc.estado_cita, jc.total_facturado, 
                jc.metodo_pago, jc.comprobante_pago, jc.notas_coordinador, jc.sexo
            FROM jornada_clientes jc 
            JOIN clientes c ON jc.cliente_id = c.id 
            WHERE jc.jornada_id = ?
        ");
        $stmt->execute([$jornada_id]);
        $pacientes = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Obtener el consumo de suministros para cada paciente
        $stmt_consumo = $conn->prepare("SELECT producto_id, cantidad_usada FROM jornada_consumo WHERE jornada_cliente_id = ?");
        foreach($pacientes as &$paciente) {
            $stmt_consumo->execute([$paciente['jornada_cliente_id']]);
            // Se añade el campo 'consumo' que el frontend necesita
            $paciente['consumo'] = $stmt_consumo->fetchAll(PDO::FETCH_ASSOC);
        }

        $response = ['success' => true, 'data' => $pacientes];
        
    } catch(PDOException $e) { 
        $response['message'] = 'Error al cargar pacientes de la jornada.'; 
        error_log($e->getMessage()); 
    }
    break;

    case 'actualizar_paciente_jornada':
        $jornada_cliente_id = filter_input(INPUT_POST, 'jornada_cliente_id', FILTER_VALIDATE_INT);
        $jornada_id = filter_input(INPUT_POST, 'jornada_id', FILTER_VALIDATE_INT);
        if (!$jornada_cliente_id || !$jornada_id) { $response['message'] = 'ID de paciente o jornada no válido.'; break; }

        try {
            $conn->beginTransaction();

            // Obtener ruta_id para las operaciones de stock
            $stmt_ruta = $conn->prepare("SELECT ruta_id FROM jornadas WHERE id = ?");
            $stmt_ruta->execute([$jornada_id]);
            $ruta_id = $stmt_ruta->fetchColumn();

            // Lógica de subida de archivo
            $comprobante_filename = null;
            if (isset($_FILES['comprobante_pago']) && $_FILES['comprobante_pago']['error'] == 0) {
                // ... (código de subida de archivo que ya tienes)
            }

            // --- GESTIÓN DE CONSUMO DE SUMINISTROS ---
            $suministros_usados = $_POST['suministros'] ?? [];

            // Obtener el consumo anterior para calcular diferencias
            $stmt_consumo_anterior = $conn->prepare("SELECT producto_id, cantidad_usada FROM jornada_consumo WHERE jornada_cliente_id = ?");
            $stmt_consumo_anterior->execute([$jornada_cliente_id]);
            $consumo_anterior = $stmt_consumo_anterior->fetchAll(PDO::FETCH_KEY_PAIR);

            // Borrar el consumo anterior para registrar el nuevo
            $conn->prepare("DELETE FROM jornada_consumo WHERE jornada_cliente_id = ?")->execute([$jornada_cliente_id]);

            $stmt_update_stock_ruta = $conn->prepare("UPDATE ruta_stock SET cantidad = cantidad + ? WHERE ruta_id = ? AND producto_id = ?");
            $stmt_insert_consumo = $conn->prepare("INSERT INTO jornada_consumo (jornada_cliente_id, producto_id, cantidad_usada) VALUES (?, ?, ?)");

            foreach ($suministros_usados as $producto_id => $cantidad_nueva) {
                $cantidad_nueva = (int)$cantidad_nueva;
                $cantidad_anterior = (int)($consumo_anterior[$producto_id] ?? 0);
                $diferencia = $cantidad_anterior - $cantidad_nueva; // Invertido: positivo si se devuelve, negativo si se consume más

                if ($diferencia != 0) {
                    $stmt_update_stock_ruta->execute([$diferencia, $ruta_id, $producto_id]);
                }

                if ($cantidad_nueva > 0) {
                    $stmt_insert_consumo->execute([$jornada_cliente_id, $producto_id, $cantidad_nueva]);
                }
            }
            
            // Actualizar datos del paciente, incluyendo las pruebas rápidas
            $sql = "UPDATE jornada_clientes SET estado_cita = ?, metodo_pago = ?, notas_coordinador = ?";
            $params = [$_POST['estado_cita'], $_POST['metodo_pago'] ?: null, $_POST['notas_coordinador']];
            
            if ($comprobante_filename) {
                $sql .= ", comprobante_pago = ?";
                $params[] = $comprobante_filename;
            }

            $sql .= " WHERE id = ?";
            $params[] = $jornada_cliente_id;

            $stmt = $conn->prepare($sql);
            $stmt->execute($params);
            
            $conn->commit();
            registrar_accion('actualiza_paciente_jornada', "Actualizó datos y consumo para el paciente-jornada ID: {$jornada_cliente_id}");
            $response = ['success' => true, 'message' => 'Información y consumo del paciente actualizados.'];

        } catch (Exception $e) {
            $conn->rollBack();
            $response['message'] = 'Error al actualizar: ' . $e->getMessage();
            error_log($e->getMessage());
        }
        break;
}

echo json_encode($response);
?>