<?php
// Agregar al inicio para ver todos los errores (remover en producción)
error_reporting(E_ALL);
ini_set('display_errors', 1);

define('IS_AJAX', true);
require_once '../administracion/includes/seguridad.php';

header('Content-Type: application/json');
$response = ['success' => false, 'message' => 'Acción no válida.'];
$usuario_id = $_SESSION['user_id'] ?? 0; // El ID del usuario logueado (doctor/admin)

$action = $_POST['action'] ?? $_GET['action'] ?? '';

switch ($action) {

    case 'buscar_afiliado_por_cedula':
    try {
        $cedula = $_GET['cedula'] ?? '';
        if (empty($cedula)) {
            throw new Exception("Debe proporcionar una cédula.");
        }

        $afiliado = null;
        $miembro_encontrado = null; // Inicializar la variable

        // 1. Primero buscar si es un titular
        $sql_titular = "SELECT id, numero_contrato, nombre_titular, apellido_titular, cedula_titular, estado FROM afiliaciones WHERE cedula_titular = ?";
        $stmt_titular = $conn->prepare($sql_titular);
        if (!$stmt_titular) {
            throw new Exception("Error preparando consulta titular: " . implode(", ", $conn->errorInfo()));
        }
        
        if (!$stmt_titular->execute([$cedula])) {
            throw new Exception("Error ejecutando consulta titular: " . implode(", ", $stmt_titular->errorInfo()));
        }
        
        $afiliado = $stmt_titular->fetch(PDO::FETCH_ASSOC);

        // 2. Si no se encontró como titular, buscar en miembros
        if (!$afiliado) {
            $sql_miembro = "SELECT am.*, a.id as afiliacion_id, a.numero_contrato, a.nombre_titular, a.apellido_titular, a.cedula_titular, a.estado 
                           FROM afiliados_miembros am 
                           JOIN afiliaciones a ON am.afiliacion_id = a.id 
                           WHERE am.cedula = ?";
            $stmt_miembro = $conn->prepare($sql_miembro);
            if (!$stmt_miembro) {
                throw new Exception("Error preparando consulta miembro: " . implode(", ", $conn->errorInfo()));
            }
            
            if (!$stmt_miembro->execute([$cedula])) {
                throw new Exception("Error ejecutando consulta miembro: " . implode(", ", $stmt_miembro->errorInfo()));
            }
            
            $miembro_data = $stmt_miembro->fetch(PDO::FETCH_ASSOC);
            
            if ($miembro_data) {
                // Si se encontró un miembro, construir la estructura del afiliado usando los datos de la afiliación
                $afiliado = [
                    'id' => $miembro_data['afiliacion_id'],
                    'numero_contrato' => $miembro_data['numero_contrato'],
                    'nombre_titular' => $miembro_data['nombre_titular'],
                    'apellido_titular' => $miembro_data['apellido_titular'],
                    'cedula_titular' => $miembro_data['cedula_titular'],
                    'estado' => $miembro_data['estado']
                ];
                
                // Guardar la información del miembro encontrado
                $miembro_encontrado = [
                    'id' => $miembro_data['id'],
                    'nombre' => $miembro_data['nombre'],
                    'apellido' => $miembro_data['apellido'],
                    'cedula' => $miembro_data['cedula'],
                    'parentesco' => $miembro_data['parentesco']
                ];
            }
        }

        // 3. Si no se encontró ni como titular ni como miembro
        if (!$afiliado) {
            $response['message'] = 'No se encontró ningún afiliado (titular o miembro) con esa cédula.';
            echo json_encode($response);
            exit;
        }
        
        // 4. Verificar que el contrato esté activo
        if ($afiliado['estado'] !== 'Activo') {
            $response['message'] = 'El contrato de este afiliado no se encuentra activo.';
            echo json_encode($response);
            exit;
        }

        // 5. Buscar el historial de visitas de toda la afiliación
        $sql_visitas = "SELECT v.id, v.fecha_visita, v.motivo_consulta, u.nombre as doctor_nombre,
                IF(v.miembro_id IS NULL, CONCAT(a.nombre_titular, ' ', a.apellido_titular, ' (Titular)'), CONCAT(am.nombre, ' ', am.apellido)) as paciente_nombre
         FROM visitas_medicas v
         JOIN usuarios u ON v.doctor_id = u.id
         JOIN afiliaciones a ON v.afiliacion_id = a.id
         LEFT JOIN afiliados_miembros am ON v.miembro_id = am.id
         WHERE v.afiliacion_id = ?
         ORDER BY v.fecha_visita DESC";
        
        $stmt_visitas = $conn->prepare($sql_visitas);
        if (!$stmt_visitas) {
            throw new Exception("Error preparando consulta visitas: " . implode(", ", $conn->errorInfo()));
        }
        
        if (!$stmt_visitas->execute([$afiliado['id']])) {
            throw new Exception("Error ejecutando consulta visitas: " . implode(", ", $stmt_visitas->errorInfo()));
        }
        
        $historial = $stmt_visitas->fetchAll(PDO::FETCH_ASSOC);

        // 6. Buscar la lista de miembros para el formulario de nueva visita
        $miembros_para_select = [];
        // Añadir al titular a la lista
        $miembros_para_select[] = ['id' => 'T_' . $afiliado['id'], 'nombre' => $afiliado['nombre_titular'] . ' ' . $afiliado['apellido_titular'] . ' (Titular)'];
        
        $sql_miembros = "SELECT id, nombre, apellido, parentesco FROM afiliados_miembros WHERE afiliacion_id = ?";
        $stmt_miembros = $conn->prepare($sql_miembros);
        if (!$stmt_miembros) {
            throw new Exception("Error preparando consulta miembros: " . implode(", ", $conn->errorInfo()));
        }
        
        if (!$stmt_miembros->execute([$afiliado['id']])) {
            throw new Exception("Error ejecutando consulta miembros: " . implode(", ", $stmt_miembros->errorInfo()));
        }
        
        while($row = $stmt_miembros->fetch(PDO::FETCH_ASSOC)) {
            $miembros_para_select[] = ['id' => 'M_' . $row['id'], 'nombre' => $row['nombre'] . ' ' . $row['apellido'] . ' (' . $row['parentesco'] . ')'];
        }

        $response = [
            'success' => true,
            'afiliado' => $afiliado,
            'historial' => $historial,
            'miembros' => $miembros_para_select,
            'miembro_buscado' => $miembro_encontrado // Ahora está definida correctamente
        ];

    } catch (Exception $e) {
        error_log("Error en buscar_afiliado_por_cedula: " . $e->getMessage());
        $response['message'] = 'Error al buscar afiliado: ' . $e->getMessage();
    }
    break;

    case 'get_visita_detalle':
        try {
            $visita_id = $_GET['id'] ?? 0;
            
            if (!puede('ver_historial_paciente')) { 
                throw new Exception("Acceso denegado."); 
            }

            // Consulta para los detalles de la visita
            $sql_visita = "SELECT v.*, u.nombre as doctor_nombre, 
                          IF(v.miembro_id IS NULL, CONCAT(a.nombre_titular, ' ', a.apellido_titular, ' (Titular)'), CONCAT(am.nombre, ' ', am.apellido)) as paciente_nombre 
                          FROM visitas_medicas v 
                          JOIN usuarios u ON v.doctor_id = u.id 
                          JOIN afiliaciones a ON v.afiliacion_id = a.id 
                          LEFT JOIN afiliados_miembros am ON v.miembro_id = am.id 
                          WHERE v.id = ?";
            
            $stmt_visita = $conn->prepare($sql_visita);
            if (!$stmt_visita) {
                throw new Exception("Error preparando consulta visita: " . implode(", ", $conn->errorInfo()));
            }
            
            if (!$stmt_visita->execute([$visita_id])) {
                throw new Exception("Error ejecutando consulta visita: " . implode(", ", $stmt_visita->errorInfo()));
            }
            
            $visita = $stmt_visita->fetch(PDO::FETCH_ASSOC);

            if (!$visita) { 
                throw new Exception("Visita no encontrada."); 
            }

            // Consulta para los suministros de esa visita
            $sql_suministros = "SELECT vs.cantidad_usada, vs.precio_registrado, ap.nombre as producto_nombre 
                               FROM visitas_suministros vs 
                               JOIN almacen_productos ap ON vs.producto_id = ap.id 
                               WHERE vs.visita_id = ?";
            
            $stmt_suministros = $conn->prepare($sql_suministros);
            if (!$stmt_suministros) {
                throw new Exception("Error preparando consulta suministros: " . implode(", ", $conn->errorInfo()));
            }
            
            if (!$stmt_suministros->execute([$visita_id])) {
                throw new Exception("Error ejecutando consulta suministros: " . implode(", ", $stmt_suministros->errorInfo()));
            }
            
            $suministros = $stmt_suministros->fetchAll(PDO::FETCH_ASSOC);

            $response = ['success' => true, 'visita' => $visita, 'suministros' => $suministros];

        } catch (Exception $e) {
            error_log("Error en get_visita_detalle: " . $e->getMessage());
            $response['message'] = 'Error al obtener detalles de la visita: ' . $e->getMessage();
        }
        break;

    case 'buscar_productos':
        try {
            $term = $_GET['term'] ?? '';
            $sql_productos = "SELECT p.id, p.nombre, s.cantidad as stock 
                             FROM almacen_productos p 
                             JOIN almacen_stock s ON p.id = s.producto_id 
                             WHERE (p.nombre LIKE ? OR p.sku LIKE ?) AND s.cantidad > 0
                             LIMIT 10";
            
            $stmt = $conn->prepare($sql_productos);
            if (!$stmt) {
                throw new Exception("Error preparando consulta productos: " . implode(", ", $conn->errorInfo()));
            }
            
            $searchTerm = "%{$term}%";
            if (!$stmt->execute([$searchTerm, $searchTerm])) {
                throw new Exception("Error ejecutando consulta productos: " . implode(", ", $stmt->errorInfo()));
            }
            
            echo json_encode($stmt->fetchAll(PDO::FETCH_ASSOC));
            exit;
            
        } catch (Exception $e) {
            error_log("Error en buscar_productos: " . $e->getMessage());
            echo json_encode(['error' => $e->getMessage()]);
            exit;
        }

    case 'guardar_visita':
        if (!puede('registrar_visita_medica')) {
            $response['message'] = 'No tienes permiso para registrar visitas.';
            break;
        }

        // Verificación de token CSRF para acciones POST
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
                $response['message'] = 'Error de seguridad (CSRF).';
                echo json_encode($response);
                exit();
            }
        }

        try {
            $conn->beginTransaction();
            
            $miembro_parts = explode('_', $_POST['miembro_id']);
            $miembro_id = ($miembro_parts[0] === 'T') ? null : (int)$miembro_parts[1];
            $doctor_id = $usuario_id;

            $sql_visita = "INSERT INTO visitas_medicas (afiliacion_id, miembro_id, doctor_id, fecha_visita, motivo_consulta, diagnostico, tratamiento, notas_medicas) VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt_visita = $conn->prepare($sql_visita);
            if (!$stmt_visita) {
                throw new Exception("Error preparando inserción visita: " . implode(", ", $conn->errorInfo()));
            }
            
            $params_visita = [
                $_POST['afiliacion_id'], 
                $miembro_id, 
                $doctor_id, 
                $_POST['fecha_visita'],
                $_POST['motivo_consulta'], 
                $_POST['diagnostico'], 
                $_POST['tratamiento'], 
                $_POST['notas_medicas']
            ];
            
            if (!$stmt_visita->execute($params_visita)) {
                throw new Exception("Error ejecutando inserción visita: " . implode(", ", $stmt_visita->errorInfo()));
            }
            
            $visita_id = $conn->lastInsertId();

            if (isset($_POST['suministros']) && is_array($_POST['suministros'])) {
                foreach ($_POST['suministros'] as $s) {
                    if (empty($s['id']) || empty($s['cantidad']) || !is_numeric($s['id']) || !is_numeric($s['cantidad'])) continue;
                    
                    // Lógica para descontar del stock
                    $sql_stock = "UPDATE almacen_stock SET cantidad = cantidad - ? WHERE producto_id = ? AND cantidad >= ?";
                    $stmt_stock = $conn->prepare($sql_stock);
                    if (!$stmt_stock) {
                        throw new Exception("Error preparando actualización stock: " . implode(", ", $conn->errorInfo()));
                    }
                    
                    if (!$stmt_stock->execute([(int)$s['cantidad'], (int)$s['id'], (int)$s['cantidad']])) {
                        throw new Exception("Error actualizando stock: " . implode(", ", $stmt_stock->errorInfo()));
                    }
                    
                    if ($stmt_stock->rowCount() == 0) {
                        throw new Exception("Stock insuficiente para el producto ID {$s['id']}.");
                    }

                    // Registrar movimiento
                    $sql_mov = "INSERT INTO almacen_movimientos (producto_id, tipo_movimiento, cantidad, motivo, referencia_id, usuario_id) VALUES (?, 'egreso', ?, ?, ?, ?)";
                    $stmt_mov = $conn->prepare($sql_mov);
                    if (!$stmt_mov) {
                        throw new Exception("Error preparando inserción movimiento: " . implode(", ", $conn->errorInfo()));
                    }
                    
                    if (!$stmt_mov->execute([(int)$s['id'], (int)$s['cantidad'], "Uso en visita medica #{$visita_id}", $visita_id, $usuario_id])) {
                        throw new Exception("Error insertando movimiento: " . implode(", ", $stmt_mov->errorInfo()));
                    }

                    // Registrar en la tabla de la visita
                    $sql_visita_s = "INSERT INTO visitas_suministros (visita_id, producto_id, cantidad_usada, precio_registrado) VALUES (?, ?, ?, 0.00)";
                    $stmt_visita_s = $conn->prepare($sql_visita_s);
                    if (!$stmt_visita_s) {
                        throw new Exception("Error preparando inserción visita_suministros: " . implode(", ", $conn->errorInfo()));
                    }
                    
                    if (!$stmt_visita_s->execute([$visita_id, (int)$s['id'], (int)$s['cantidad']])) {
                        throw new Exception("Error insertando visita_suministros: " . implode(", ", $stmt_visita_s->errorInfo()));
                    }
                }
            }

            $conn->commit();
            registrar_accion('registrar_visita', "Registró nueva visita médica ID: {$visita_id}");
            $response = ['success' => true, 'message' => 'Visita médica registrada exitosamente.'];

        } catch (Exception $e) {
            $conn->rollBack();
            error_log("Error al guardar visita: " . $e->getMessage());
            $response['message'] = 'Error al guardar la visita: ' . $e->getMessage();
        }
        break;

    default:
        $response['message'] = 'Acción desconocida en visitas_ajax: ' . $action;
        break;
}

echo json_encode($response);